#uses "dpGroups.ctl"
#uses "hosts.ctl"

// limit for the CPU-usage
float f_limitCPU;

// store the current threadIDs
dyn_int di_simThreadIDs;

// store the current number of threads
int i_threadNum;

// dyn-variables for the simulated dp-elements
dyn_string ds_dps, ds_simulatedDPs;

// index of next simulated DPE
int i_simuDPEIndex = 1;

// variable to check if simulation is already active and which simulation was used
bool b_started;
int i_simType;

// store the information if the script is running on host1 or host2 in a redundant system
int i_hostNum;
string s_reduPostFix;


// store the simulation settings for dpConnect
anytype at_globalSimSettings;

// mapping-variables for the current value and the direction
// the name of the DPE is used as the key
mapping m_currentValue;
mapping m_direction;


main()
{
  i_hostNum = initHosts();

  // get the information if the script is running on host1 or host2 in a redundant system
  if(isRedundant())
  {
    i_hostNum = initHosts();
    
    if(i_hostNum == 2)
      s_reduPostFix = "_2";
  }
  
  // check if the datapoint NFR_messageSimulation exists and if the settings are correct
  // incorrect settings are set with default values
  if(!dpExists("NFR_messageSimulation"))
  {
    dyn_errClass deC_error;
    string s_error;
    
    //DebugTN(__LINE__,"Datapoint NFR_messageSimulation is missing - message simulation cannot be used");
    s_error = getCatStr("_errors","00007");
    deC_error = makeError("_errors",PRIO_FATAL,ERR_CONTROL,54,"NFR_messageSimulation " + s_error);

    throwError(deC_error);
  }
  else
  {
    float f_CPU_limit, f_rangeMin, f_rangeMax, f_increment;
    int i_DPE_count, i_MSG_count, i_MSG_interval;
    bool b_active, b_startValue;
    int i_VC, i_typeVC, i_typeSIM, i_typeVCBit;
    string s_dpGroup;
    langString ls_groupName;
    dyn_langString dls_groupNames;

    dpGet("NFR_messageSimulation.settings.CPU_limit",f_CPU_limit,
          "NFR_messageSimulation.settings.DPE_count",i_DPE_count,
          "NFR_messageSimulation.settings.MSG_count",i_MSG_count,
          "NFR_messageSimulation.settings.DP_group",s_dpGroup,
          "NFR_messageSimulation.settings.Range_min",f_rangeMin,
          "NFR_messageSimulation.settings.Range_max",f_rangeMax,
          "NFR_messageSimulation.settings.Increment",f_increment,
          "NFR_messageSimulation.settings.VC_type",i_typeVC,
          "NFR_messageSimulation.settings.SIM_type",i_typeSIM,
          "NFR_messageSimulation.settings.Start_value",b_startValue,
          "NFR_messageSimulation.settings.VC_typeBit",i_typeVCBit,
          "NFR_messageSimulation.settings.MSG_interval",i_MSG_interval,
          "NFR_messageSimulation.control.active",b_active);

    // get the current list of datapoint groups
    dls_groupNames = groupGetNames();

    ls_groupName = groupDpNameToName(s_dpGroup);

    // check if the settings at the datapoint are correct - otherwise deactivate the simulation
    if(!dynContains(dls_groupNames,ls_groupName))
    {
      // if configured group does not exist set the simulation inactive
      b_active = false;
    }

    // set initial values if nothing was defined
    if(f_rangeMin == 0 && f_rangeMax == 0)
    {
      f_rangeMin = 1;
      f_rangeMax = 10;
    
      // deactivate simulation
      b_active = false;
    }

    if(f_increment == 0)
    {
      f_increment = 1;

      // deactivate simulation
      b_active = false;
    }
  
    if(i_MSG_interval == 0)
    {
      i_MSG_interval == 5;

      // deactivate simulation
      b_active = false;
    }
  
    if(i_typeSIM == 0 || i_typeSIM > 2)
    {
      i_typeSIM = 1;

      // deactivate simulation
      b_active = false;
    }

    if(i_typeVC == 0 || i_typeVC > 3)
    {
      i_typeVC = 1;

      // deactivate simulation
      b_active = false;
    }

    if(i_typeVCBit == 0 || i_typeVCBit > 4)
    {
      i_typeVCBit = 1;

      // deactivate simulation
      b_active = false;
    }

    // write the modified settings to the datapoint
    dpSetWait("NFR_messageSimulation.settings.CPU_limit",f_CPU_limit,
              "NFR_messageSimulation.settings.DPE_count",i_DPE_count,
              "NFR_messageSimulation.settings.MSG_count",i_MSG_count,
              "NFR_messageSimulation.settings.DP_group",s_dpGroup,
              "NFR_messageSimulation.settings.Range_min",f_rangeMin,
              "NFR_messageSimulation.settings.Range_max",f_rangeMax,
              "NFR_messageSimulation.settings.Increment",f_increment,
              "NFR_messageSimulation.settings.VC_type",i_typeVC,
              "NFR_messageSimulation.settings.SIM_type",i_typeSIM,
              "NFR_messageSimulation.settings.Start_value",b_startValue,
              "NFR_messageSimulation.settings.VC_typeBit",i_typeVCBit,
              "NFR_messageSimulation.settings.MSG_interval",i_MSG_interval,
              "NFR_messageSimulation.control.active",b_active);
  }
  
  dpConnect("NFR_messageSimulation",0,"NFR_messageSimulation.settings.CPU_limit",
                                      "NFR_messageSimulation.settings.DPE_count",
                                      "NFR_messageSimulation.settings.MSG_count",
                                      "NFR_messageSimulation.settings.DP_group",
                                      "NFR_messageSimulation.settings.Range_min",
                                      "NFR_messageSimulation.settings.Range_max",
                                      "NFR_messageSimulation.settings.Increment",
                                      "NFR_messageSimulation.settings.VC_type",
                                      "NFR_messageSimulation.settings.SIM_type",
                                      "NFR_messageSimulation.settings.Start_value",
                                      "NFR_messageSimulation.settings.VC_typeBit",
                                      "NFR_messageSimulation.settings.MSG_interval",
                                      "NFR_messageSimulation.control.active");
}

NFR_messageSimulation(string s_dpCPULimit, float f_CPULimit,
                      string s_dpDPECount, int i_DPECount,
                      string s_dpMSGCount, int i_msgCount,
                      string s_dpDPGroup, string s_groupDp,
                      string s_dpRangeMin, float f_rangeMin,
                      string s_dpRangeMax, float f_rangeMax,
                      string s_dpIncrement, float f_increment,
                      string s_dpVCType, int i_typeVC,
                      string s_dpSIMType, int i_typeSIM,
                      string s_dpStartValue, bool b_startValue,
                      string s_dpVCTypeBit, int i_typeVCBit,
                      string s_dpMSGinterval, int i_msgInterval,
                      string s_dpActive, bool b_active)
{
  int i_return;
  dyn_anytype da_settings;
  anytype at_simSettings;
  
  int i;
  
  da_settings[1] = i_DPECount;
  da_settings[2] = f_rangeMin;
  da_settings[3] = f_rangeMax;
  da_settings[4] = f_increment;
  da_settings[5] = i_typeVC;
  da_settings[6] = i_typeVCBit;
  
  at_simSettings = da_settings;
  

  // check if simulation is active and if it is started already
  if(b_active == 1 && b_started == 0)
  {
    dyn_errClass deC_error;
    string s_error;

    s_groupDp = dpSubStr(s_groupDp,DPSUB_DP);
    
    if(s_groupDp == "")
    {
      //DebugTN(__LINE__,"datapoint-group not defined - simulation inactive");
      
      s_error = getCatStr("WinCC_OA_tools","noDpGroup");
      deC_error = makeError("_errors",PRIO_WARNING,ERR_PARAM,54,s_error);

      throwError(deC_error);
      dpSetWait("NFR_messageSimulation.control.active",0);
      return;
    }
    else
    {
      dynClear(ds_dps);
      
      i_return = NFR_resolveDPGroup(s_groupDp);
      
      if(i_return == -1)
      {
        //DebugTN(__LINE__,"no dp-elements for simulation defined!");
        s_error = getCatStr("WinCC_OA_tools","noDPEinGroup");
        deC_error = makeError("_errors",PRIO_WARNING,ERR_PARAM,54,s_error);

        throwError(deC_error);
        dpSetWait("NFR_messageSimulation.control.active",0);
        return;
      }
    }
   
    f_limitCPU = f_CPULimit;
    
    // check which simulation type was selected
    // check for the CPU-usage
    if(i_typeSIM == 1)
    {
      string s_version;
      
      s_version = VERSION;
      
      at_globalSimSettings = at_simSettings;

      // connect to the dp-element for the current CPU-usage of the event-manager
      if(s_version == "3.9" || s_version == "3.10")
      {
        dpConnect("NFR_simulationCheckCPU",0,"PVSS00event" + s_reduPostFix + ".CPU_Percent");
      }
      else
      {
        dpConnect("NFR_simulationCheckCPU",0,"WCCILevent" + s_reduPostFix + ".CPU_Percent");
      }
        
      i_simType = i_typeSIM;
      b_started = 1;
    }
    
    // check for the number of messages (not value changes)
    else if(i_typeSIM == 2)
    {
      at_globalSimSettings = at_simSettings;
      
      // start the given number of simulation threads
      NFR_startSimuThreads(i_msgCount, at_simSettings, b_startValue,i_msgInterval);
      i_simType = i_typeSIM;
      b_started = 1;

      // set the information-DPE
      dpSet("NFR_messageSimulation.information.Num_threads",i_msgCount,
            "NFR_messageSimulation.information.VC_per_sec",i_msgCount * i_DPECount / i_msgInterval);
    }
  }
  else if(b_started == 1)
  {
    int i;
    
    // do the dpDisconnect and stop the simulation-threads
    if(i_simType == 1)
    {
      string s_version;
      
      s_version = VERSION;
      
      if(s_version == "3.9" || s_version == "3.10")
        dpDisconnect("NFR_simulationCheckCPU","PVSS00event" + s_reduPostFix + ".CPU_Percent");
      else
        dpDisconnect("NFR_simulationCheckCPU","WCCILevent" + s_reduPostFix + ".CPU_Percent");
    }
    
    for(i=1;i<=dynlen(di_simThreadIDs);i++)
    {
      stopThread(di_simThreadIDs[i]);
    }
    
    // reset the global variables
    dynClear(di_simThreadIDs);
    i_threadNum = 0;
    i_simuDPEIndex = 1;
    i_simType = 0;
    b_started = 0;
    
    // reset the mapping information
    mappingClear(m_currentValue);
    mappingClear(m_direction);
    
    // set the information-DPE
    dpSet("NFR_messageSimulation.information.Num_threads",0,
          "NFR_messageSimulation.information.VC_per_sec",0);
  }
}

NFR_startSimuThreads(int i_threads, anytype at_simSettings, bool b_startValue, int i_msgInterval)
{
  // number of dp-elements set in one dpSet
  int i_numDPS;
  
  int i_nextThreadNum;
  int i;
  dyn_anytype da_settings;

  da_settings = at_globalSimSettings;
  
  i_numDPS = da_settings[1];

  for(i=1;i<=i_threads;i++)
  {
    dyn_string ds_dps_sim;

    // calculated number of simulated DPEs
    int i_numSimulatedDPEs;

    i_threadNum = dynlen(di_simThreadIDs);
    i_nextThreadNum = i_threadNum+1;

    i_numSimulatedDPEs = i_nextThreadNum * i_numDPS;
  
    while(dynlen(ds_simulatedDPs) < i_numSimulatedDPEs)
    {
      dyn_string ds_addDPs;
    
      //DebugTN(__LINE__,"append ds_dps to ds_simulatedDPs","i_numSimulatedDPEs",i_numSimulatedDPEs,"dynlen(ds_simulatedDPs)",dynlen(ds_simulatedDPs));
    
      ds_addDPs = ds_dps;
      dynAppend(ds_simulatedDPs,ds_addDPs);
    }

    while(dynlen(ds_dps_sim) < i_numDPS)
    {
      dynAppend(ds_dps_sim,ds_simulatedDPs[i_simuDPEIndex] + ":_original.._value");
      i_simuDPEIndex++;
    }
  
    if(dynlen(ds_dps_sim) == i_numDPS)
    {
      time t_start;
      int i_simThreadID;
        
      t_start = getCurrentTime();
      //DebugTN("start thread");
      
      i_simThreadID = startThread("NFR_simuThread_values",ds_dps_sim,t_start,at_simSettings,b_startValue,i_msgInterval);
        
      dynAppend(di_simThreadIDs,i_simThreadID);

      i_threadNum = dynlen(di_simThreadIDs);
      //DebugTN(__LINE__,"thread started",i_threadNum);
      //DebugTN(__LINE__,"di_simThreadIDs",di_simThreadIDs);
    }
  }
}


NFR_simulationCheckCPU(string s_dpCPU, float f_currentCPU)
{
  float f_diffCPU;
  float f_factor;
  int i_newThreadNum;
  int i;
  int i_numDPS;
  dyn_anytype da_settings;
  
  da_settings = at_globalSimSettings;
  
  i_numDPS = da_settings[1];
  
  // if the current CPU usage is 0 set a default value
  if(f_currentCPU == 0)
    f_currentCPU = 1;
  
  // calculate the difference to avoid that new threads are started or stopped when the difference between
  // the currentCPU and the limit is below a given threshold
  f_diffCPU = f_currentCPU - f_limitCPU;
  //DebugTN(__LINE__,"f_diffCPU",f_diffCPU);
  
  if(f_diffCPU < 0)
    f_diffCPU = f_diffCPU * -1;
  //f_diffCPU = abs(f_diffCPU);

  //DebugTN(__LINE__,"absolute f_diffCPU",f_diffCPU);

  // calculate the factor to add/remove threads
  f_factor = f_limitCPU/f_currentCPU;

  // limit the factor to avoid an overload situation  
  if(f_factor > 10)
    f_factor = 10;
  
  if(i_threadNum == 0)
    i_threadNum = 1;
  
  i_newThreadNum = i_threadNum * f_factor;

  //DebugTN(__LINE__,"f_currentCPU",f_currentCPU,"f_limitCPU",f_limitCPU,"f_diffCPU",f_diffCPU);
  //DebugTN(__LINE__,"i_threadNum",i_threadNum,"i_newThreadNum",i_newThreadNum,"f_factor",f_factor);
  
  if(f_currentCPU > f_limitCPU && f_diffCPU > 2)
  {
    // stop simulation threads and set the index for the next simulated DPE
    for(i=i_threadNum;i>=i_newThreadNum;i--)
    {
      //DebugTN(__LINE__,"stop thread with ID",di_simThreadIDs[i],"dynlen(di_simThreadIDs)",dynlen(di_simThreadIDs));
      stopThread(di_simThreadIDs[i]);
      dynRemove(di_simThreadIDs,dynlen(di_simThreadIDs));
      i_simuDPEIndex = i_simuDPEIndex - i_numDPS;
    }
  }
  else if(f_currentCPU < f_limitCPU && f_diffCPU > 2)
  {
    // start new simulation threads
    for(i=i_threadNum;i<=i_newThreadNum;i++)
    {
      //DebugTN(__LINE__,"call NFR_startSimuThreads i_threadNum",i_threadNum+1);
      NFR_startSimuThreads(1,at_globalSimSettings, 1);
    }
  }
  i_threadNum=dynlen(di_simThreadIDs);
 
  // set the information-DPE
  dpSet("NFR_messageSimulation.information.Num_threads",i_threadNum,
        "NFR_messageSimulation.information.VC_per_sec",i_threadNum * i_numDPS / i_msgInterval);

}


NFR_simuThread_values(dyn_string ds_dpl, time t_start, anytype at_simSettings, bool b_startValue, int i_msgInterval)
{
  dyn_anytype da_vals, da_settings;
  int i;
  bool b_val, b_up;
  float f_val;
  int i_delayms;
  int i_delaysec;
  time t_next;
  time t_current;
  time t_initDelay;
  float f_randTime;

  int i_typeVC, i_typeVCBit;
  float f_rangeMin, f_rangeMax, f_increment;

  // calculate an initial delay and add it to the start time t_next for the next simulation to avoid peaks in the simulation
  f_randTime = rand();
  t_initDelay = (f_randTime/32767) * i_msgInterval;
  
  //DebugTN(__LINE__,"NFR_simuThread_values - t_initDelay",t_initDelay);
  
  t_next = t_start + t_initDelay;

  //DebugTN(__LINE__,"NFR_simuThread_values - t_next, t_start",t_next,t_start);
  
  
  // read the settings for the simulation
  da_settings = at_simSettings;
  
  f_rangeMin = da_settings[2];
  f_rangeMax = da_settings[3];
  f_increment = da_settings[4];
  i_typeVC = da_settings[5];
  i_typeVCBit = da_settings[6];

  
  // start with increasing values
  b_up = 1;
  
  // set the minimum start value
  f_val = f_rangeMin;
  
  while(1)
  {
    dyn_errClass err;

    for(i=1;i<=dynlen(ds_dpl);i++)
    {
      string s_dpe;
      
      s_dpe = ds_dpl[i];
      
      //DebugTN(__LINE__,"NFR_simuThread_values - dp-element + datatype",ds_dpl[i],dpElementType(ds_dpl[i]));
      if(dpElementType(s_dpe) == DPEL_BOOL)
      {
        // check which simulation type was defined for boolean values
        // toggling the values is made after the for-loop for the dp-elements
        if(i_typeVCBit == 2)
          b_val = 0;
        else if(i_typeVCBit == 3)
          b_val = 1;
        else if(i_typeVCBit == 4)
        {
          float f_randValue;
      
          f_randValue = rand();
          
          // if the value is lower than the 32767/2 the value 0 otherwise 1 is set
          if(f_randValue < 32767/2)
            b_val = 0;
          else
            b_val = 1;
        }
        //DebugTN(__LINE__,"NFR_simuThread_values - append value",ds_dpl[i],b_val);
        dynAppend(da_vals,b_val);
      }
      else
      {
        // calculate the value for the ramp simulation
        if(i_typeVC == 1)
        {
          // check if the mapping has already a value for the DPE
          if(!mappingHasKey(m_currentValue,s_dpe))
          {
            // check if a random start value shall be used
            if(b_startValue == 1)
            {
              float f_randValue;
              
              f_randValue = rand();
              f_val = (f_randValue/32767) * f_rangeMax;
            }
            // set the minimum range value as start value
            else
            {
              f_val = f_rangeMin;
            }
          }

          // get the current value and direction information from the mapping
          else
          {
            f_val = m_currentValue[s_dpe];
            b_up = m_direction[s_dpe];
            
            // make the calculation for the next value
            if(b_up == 1)
            {
              if((f_val + f_increment) <= f_rangeMax)
              {
                f_val = f_val + f_increment;
              }
              else
              {
                f_val = f_val - f_increment;
                b_up = 0;
              }
            }
            else
            {
              if((f_val - f_increment) >= f_rangeMin)
              {
                f_val = f_val - f_increment;
              }
              else
              {
                f_val = f_val + f_increment;
                b_up = 1;
              }
            }
          }
          // write the current value and direction information to the mapping
          m_currentValue[s_dpe] = f_val;
          m_direction[s_dpe] = b_up;
        }

        // calculate the value for the saw tooth simulation
        if(i_typeVC == 2)
        {
          // check if the mapping has already a value for the DPE
          if(!mappingHasKey(m_currentValue,s_dpe))
          {
            // check if a random start value shall be used
            if(b_startValue == 1)
            {
              float f_randValue;
              
              f_randValue = rand();
              f_val = (f_randValue/32767) * f_rangeMax;
            }
            // set the minimum range value as start value
            else
            {
              f_val = f_rangeMin;
            }
          }
          // get the current value and direction information from the mapping
          else
          {
            f_val = m_currentValue[s_dpe];

            // make the calculation for the next value
            if((f_val + f_increment) <= f_rangeMax)
              f_val = f_val + f_increment;
            else
              f_val = f_val + f_increment - f_rangeMax;
          }
          // write the current value to the mapping
          m_currentValue[s_dpe] = f_val;
        }
        
        //get a random value
        if(i_typeVC == 3)
        {
          float f_randValue;
      
          f_randValue = rand();
      
          f_val = (f_randValue/32767) * f_rangeMax;
        }
        //DebugTN(__LINE__,"NFR_simuThread_values - append value",ds_dpl[i],f_val);
        dynAppend(da_vals,f_val);
      }
    }

    //DebugTN("dpSet - threadID",getThreadId());
    
    // check if the array has the correct length, otherwise the dpSet is not made
    if(dynlen(ds_dpl) == dynlen(da_vals))
      dpSet(ds_dpl,da_vals);

    err = getLastError();
    if(dynlen(err) > 0)
      throwError(err);

    dynClear(da_vals);

    if(b_val == true)
      b_val = false;
    else
      b_val = true;
    
    // calculate the necessary delay before the next loop is started
    t_next = t_next + i_msgInterval;
    t_current = getCurrentTime();
    i_delaysec = second(t_next - t_current);
    i_delayms = milliSecond(t_next - t_current);
    
    //DebugTN("t_next - delaysec - delayms - threadID",t_next,i_delaysec,i_delayms,getThreadId());
    delay(i_delaysec,i_delayms);
  }
}

int NFR_resolveDPGroup(string s_groupDp)
{
  dyn_dyn_anytype dda_qres;
  dyn_errClass err;
  dyn_string ds_dplist;
  int i;
  
  dpQuery("SELECT '_online.._value' FROM 'DPGROUP(" + s_groupDp + ")' WHERE _LEAF",dda_qres);
  err = getLastError();
  if(dynlen(err) > 0)
  {
    throwError(err);
    return(-1);
  }
  else
  {
    dynRemove(dda_qres,1);
    
    for(i=1;i<=dynlen(dda_qres);i++)
    {
      string s_dp;
      
      s_dp = dpSubStr(dda_qres[i][1],DPSUB_DP_EL);
      
      // only boolean or analog values are simulated
      if(dpElementType(s_dp) == DPEL_BOOL || dpElementType(s_dp) == DPEL_INT || dpElementType(s_dp) == DPEL_FLOAT || dpElementType(s_dp) == DPEL_UINT)
        dynAppend(ds_dps,s_dp);
    }
  }

  if(dynlen(ds_dps) == 0)
  {
    return(-1);
  }
  // work-around for my tests
  //ds_dps = dpNames("NFR*.in.*","NFR_base");
  
  //DebugTN(__LINE__,"number of dp-elements for simulation",dynlen(ds_dps));
  return(1);
}
