// ***********************************
// sdSymbolDynamics.ctl
// ***********************************
//
// This library contains various helper functions that should make it 
// much easier to build rich animated symbols.
// The helper functions implement several possible animations like:
//
// - Scale
// - Move
// - Set a color
// - Show a value
//
// The following helper functions are available:
//
// - sdMove()
//   Move s ahape along a given polygon. Takes a value and a range
//   to determine the position on the polygon.
//   The polygon describes the 'path' of the shape to be moved
//
// - sdScaleVertical()
// - sdScaleHorizontal()
//   Scale a shape in x or y dkirection given a value and a range.
//
// - sdSetColorAndShow
//   Will assign a color to a shape and will then make the shape visible.
//   Typically when you're making s shape that should indicate
//   an alarm color.
//   We hide the shape when there is no alafrm color
//   ( e.g. the color is an empty string )
//
// **************************************

#uses "sdScriptWizard.ctl"

// **************************************
// Popup implementation IM 107266 / 109984 made by redwerk

/* config */
const int LISTENER_THREAD_DELAY_MS = 10; //ms to sleep in thread polling for events for popup

/* consts */
const int POPUP_ACTION_IDLE = 0;
const int POPUP_ACTION_APPLY = 1;
const int POPUP_ACTION_CLOSE = 2;

/* Global variables */
/* plain script-global variables don't fit because they are blanked for each new panel opened, 
  thus it is not possible to keep some common data for different panels */
global dyn_string sdpopupCallbackFcnList = makeDynString(); //names of the callback functions
global dyn_string sdpopupPanelNamesList = makeDynString(); //names of the popup panels 
global mapping sdpopupEventData;  //used to send events for thread popupEventListener 

// ============================================================================
void sdApplyPopup(string action, dyn_float resultFloat, dyn_string resultText)
{  
  sendEventToListener(POPUP_ACTION_APPLY, action, resultFloat, resultText);  
}

// ============================================================================
void sdClosePopup(string action, dyn_float resultFloat, dyn_string resultText)
{
  sendEventToListener(POPUP_ACTION_CLOSE, action, resultFloat, resultText);
}

// ============================================================================
void sdOpenPopup(string ReturnCallbackFunctionName,
                 string panelName, // file name of XML panel
                 string panelCaption, // title of window title bar
                 int x = -1, // position of popup. If < 0 popup is 
                 int y = -1, // centered.
                 int posMode = 0,  // 0 - centered, 1 - position as x/y, 2 - relative to object
                 dyn_string dollarParameters, // dollar parameters
                 bool bModal = true //modal or not
                     )
{
  if (isPanelOpen(panelCaption)) 
  {
    DebugN("Panel with name " + panelCaption + " is already opened");
    return; //do nothing: ChildPanelOn* will not open panel if it is already opened
  }
  if ( (x < 0 || y < 0 || posMode == 0) && bModal )
  { /* center panel */
    ChildPanelOnCentralModal(panelName, panelCaption, dollarParameters);
  }
  else if ( bModal )
  {
    ChildPanelOnModal(panelName, panelCaption, dollarParameters, x, y);
  }  
  else if ( x < 0 || y < 0 || posMode == 0)
  { /* center panel */
    ChildPanelOnCentral(panelName, panelCaption, dollarParameters);
  }
  else
  {
    if ( posMode == 2 )
      ChildPanelOnRelativ(panelName, panelCaption, dollarParameters, x, y);
    else
      ChildPanelOn(panelName, panelCaption, dollarParameters, x, y);
  }

  //wait for panel to be actually open  
  while(!isPanelOpen(panelCaption)) 
  {
    delay(0, LISTENER_THREAD_DELAY_MS);
  }
  synchronized(sdpopupEventData) 
  {
    dynAppend(sdpopupPanelNamesList, panelCaption);
    dynAppend(sdpopupCallbackFcnList, ReturnCallbackFunctionName);
    sdpopupEventData["event"] = POPUP_ACTION_IDLE;
    startThread("popupEventListener", dynlen(sdpopupPanelNamesList));
  }
}

/* ============================================================================
  Thread polling for events for popup window. 
  serves purpose of binding visibility scope of library functions to scope of opened panel
  without this thread spawned for each opened panel all callbacks are searched in root panel, 
  i.e. if one popup opens another one, callback functions for both will be searched in root panel.
  also need this to detect panel closing via standard close button "X" (seems this also can be done by dpe callback registering)
*/
private void popupEventListener(int myPopupIndex) 
{
  int popupIndex, popupEvent;

  while(true) 
  {
    synchronized(sdpopupEventData) 
    {
      popupIndex = dynlen(sdpopupPanelNamesList);
      popupEvent = sdpopupEventData["event"];
      if(popupIndex == myPopupIndex) 
      {
        if(popupEvent == POPUP_ACTION_APPLY) 
        {
          invokeCallbackFcn(sdpopupCallbackFcnList[popupIndex], sdpopupEventData["action"], sdpopupEventData["resFloat"], sdpopupEventData["resTxt"]);        
          sdpopupEventData["event"] = POPUP_ACTION_IDLE;
        } 
        else if (popupEvent == POPUP_ACTION_CLOSE) 
        {        
          dyn_anytype da, daa;
          da[1]  = myModuleName();     
          da[2]  = sdpopupPanelNamesList[dynlen(sdpopupPanelNamesList)];
          da[3] = daa;
          panelOff(da);
          //wait while panel is closed
          while(isPanelOpen(sdpopupPanelNamesList[popupIndex])) 
        {
            delay(0, LISTENER_THREAD_DELAY_MS);
          }
          invokeCallbackFcn(sdpopupCallbackFcnList[popupIndex], sdpopupEventData["action"], sdpopupEventData["resFloat"], sdpopupEventData["resTxt"]);        
          dynRemove(sdpopupCallbackFcnList, popupIndex);  
          dynRemove(sdpopupPanelNamesList, popupIndex);         
          sdpopupEventData["event"] = POPUP_ACTION_IDLE;
          return;
        }            
        if(!isPanelOpen(sdpopupPanelNamesList[popupIndex])) 
        {
          invokeCallbackFcn(sdpopupCallbackFcnList[popupIndex], "Close", makeDynFloat(), makeDynString());  
          dynRemove(sdpopupCallbackFcnList, popupIndex);    
          dynRemove(sdpopupPanelNamesList, popupIndex);   
          return;        
        }    
      }    
    }
    delay(0, LISTENER_THREAD_DELAY_MS);
  }
}


// ============================================================================
private void sendEventToListener(int event, string action, dyn_float resultFloat, dyn_string resultText) synchronized(sdpopupEventData) 
{
  sdpopupEventData["event"] = event;
  sdpopupEventData["action"] = action;
  sdpopupEventData["resFloat"] = resultFloat;
  sdpopupEventData["resTxt"] = resultText;
}


// ============================================================================
private void invokeCallbackFcn(string callbackName, string action, dyn_float resultFloat, dyn_string resultText) 
{  
   if(callbackName == "") 
   {
     return;
   }
   /* can't use callFunction here because visibility scope issues */
   execScript(
    "int main(string action, dyn_float resultFloat, dyn_string resultText) {" + 
      callbackName + "(action, resultFloat, resultText);" +
     "}" , makeDynString(""), action, resultFloat, resultText);    
}


// ============================================================================
// sdMove(). Move a shape along a given polygon
//
// Parameters:
// -  string shapeName,      The name of the shape you want to mve
// -  string refShapeName    The 'helper' polygon along which we move
// -  float value,           The value to determine new position
// -  float min,             Minimum range
// -  float max              Maximum range
//
// Usage:
//   sdMove( 
//    "MyShape",
//    "helper_line",
//    min,                // Minimum of range
//    max,                // Maximum of range
//    value );            // Value to determine position of shape
//
// ============================================================================

// implemented also in .js library for UL
void sdMove(
  string shapeName,      // The name of the shape you want to move
  string refShapeName,   // The shape ( polygon) along which we 
  float value,           // The value to determine new position   
  float min,             // minimum range
  float max,             // maximum range
  bool bUsePvRange = false,// pvrange used
  bool bDrawDirection = true// draw direction                
)
{
 
  // We'll receive the location info from the polygon
  int x;         // x coordinate of the position we request from the polygon
  int y;         // y coordinate of the position we request from the polygon
  float angle;   // angle of the polygon at the position that we request ( we don't need this )
  
  // Determine the percentage along the polygon given value and range
  float percentage = ((value-min) / (max-min)) * 100.0;

 
  // Make sure we don't exceed the 0.0 or 100.0%
  if( percentage > 100.0 ) percentage = 100.0;
  if( percentage < 0.0   ) percentage = 0.0;

  if ( !bDrawDirection )
    percentage = 100 - percentage;
  
  getValue( refShapeName, "positionOnPath", percentage, x, y, angle );


  // Now actually move the shape
  setValue( shapeName, "position", x, y );
}

// ============================================================================
// sdSetColorAndShow(). Assign a color to a shape and make shape visible
//
// Parameters:
// -  string shapeName,      The name of the shape you want to scale
// -  bool foreCol,          True when you want to change foreground color
// -  bool backCol,          True when you want to change background color
// -  string color           The color to assign
//
// Usage:
//   sdSetColorAndShow( 
//     "alarmindicator", 
//     1,                    // TRue indicates that you want to set fore ground color
//     0,                    // True indicates you want to set backgriund color
//     alarmColor );         // The color that is assigned
//
// ============================================================================

// implemented also in .js library for UL
void sdSetColorAndShow(
  string shapeName,      // The name of the shape or which you want to change the color
  bool foreCol,          // True when you want to change foreground color
  bool backCol,          // True when you want to change background color
  string color           // The color to assign
)
{
  if( foreCol ) setValue( shapeName, "foreCol", color );
  if( backCol ) setValue( shapeName, "backCol", color );
  
  // When we have a valid color name then show the symbol
  // (Typically uefull in alarm indicators. Only show
  // them when there is a valid color name )   
  setValue( shapeName, "visible", strlen( color ));
}

// ============================================================================
// sdScale(). Scale an object according to given value and range.
//
// Parameters:
// -  string shapeName,      The name of the shape you want to scale
// -  float value,           The value to calculate scaling factor
// -  float min,             Minimum range
// -  float max              Maximum range
//
// Usage:
//   sdScaleVertical( 
//     "MyShape",         // The name of the shape 
//     12.5,              // the value used to determine scaling
//     0.0,               // min of range
//     100.0 );           // max of range
//
// ============================================================================

// implemented also in .js library for UL
void sdScaleVertical(
  string shapeName,      // The name of the shape you want to scale
  float value,           // The value to calculate scaling factor
  float min,             // minimum range
  float max,             // maximum range
  bool bUsePvRange = false,// pvrange used
  float minScale = 0,
  float maxScale = 1
)
{
  // Make sure we have no divide by 0  
  if( max - min == 0.0 )
    return;
  
  // Determine the scale ( between 0.0 and 1.0) given value and range
  float scale = ( 1.0 * (maxScale - minScale) / (max - min)) * (value - min) + minScale;
  
  if (scale > maxScale) 
    scale = maxScale;
  else if (scale < minScale) 
    scale = minScale;
  
  // Make sure we don't exceed the 0.0 or 1.0
  if( scale > 1.0 ) scale = 1.0;
  if( scale < 0.0 ) scale = 0.0;
  
  //get actual scale factor
  float oldX, oldY;
  getValue("", "scale", oldX, oldY);
  
  // Now do the actual vertical scaling
  setValue( shapeName, "scale", oldX, scale );
}

// implemented also in .js library for UL
void sdScaleHorizontal(
  string shapeName,      // The name of the shape you want to scale
  float value,           // The value to calculate scaling factor
  float min,             // minimum range
  float max,             // maximum range
  bool bUsePvRange = false,// pvrange used
  float minScale = 0,
  float maxScale = 1
)
{  
  // Make sure we have no divide by 0  
  if( max - min == 0.0 )
    return;
  
  // Determine the scale ( between 0.0 and 1.0) given value and range
  float scale = ( 1.0 * (maxScale - minScale) / (max - min)) * (value - min) + minScale;
  
  if (scale > maxScale) 
    scale = maxScale;
  else if (scale < minScale) 
    scale = minScale;
  
  // Make sure we don't exceed the 0.0 or 1.0
  if( scale > 1.0 ) scale = 1.0;
  if( scale < 0.0 ) scale = 0.0;
  
  //get actual scale factor
  float oldX, oldY;
  getValue("", "scale", oldX, oldY);
  
  // Now do the actual vertical scaling
  setValue( shapeName, "scale", scale, oldY );
}

// implemented also in .js library for UL
void sdShowBadValue( 
  string shapeName,      // The name of the shape or which you want to change the color
  int mode
)
{
  if( (mode & 1) ) setValue( shapeName, "foreCol", "_invalid" );
  if( (mode & 2) ) setValue( shapeName, "backCol", "_invalid" );

}  

// implemented also in .js library for UL
void sdResetShowBadValue( 
  string shapeName,      // The name of the shape or which you want to change the color
  int mode
)
{
  if( (mode & 1) ) setValue( shapeName, "foreCol", "" );
  if( (mode & 2) ) setValue( shapeName, "backCol", "" );

}  

// implemented also in .js library for UL
void sdShowValue(
  string shapeName,    
  string dp,               // Datapoint to use to format a value    
  anytype value,
  int   formatChoice,      // 0 = no format, 1 = from dp, 2 = fixed,
  string format,
  int unitChoice,          // 0 = no unit, 1 = from dp, 2 = fixed
  string unit
)
{
  string result;
  
  // Format the floating point value
  switch( formatChoice )
  {
    case 0 : result = value; break;
    case 1 : result = dpValToString( dp, value );   break;
    case 2 : result = sdFormatValue( value, format ); break;
      
    default:
      // When no format is given, then just let PVSS do the conversion.
      result = value;
      break;
  }
  
  switch( unitChoice )
  {
    case 0 :         // 0 means, no unit
      break;         // so we're done here
      
    case 1 :         // 1 means unit from DP
      result += " " + dpGetUnit( dp );
      break;  
      
    case 2 :         // 2 means unit given here
      result += " " + unit;
      break;
  }
  
  setValue( shapeName, "text", result );
  
}


// *********************************
// sdFormatValue(). Format a float value according to a given value
//
// Parameters:
// - float f          The float that we want to turn into a string
// - string format    The format. SOmething like '+99.99'
//
// Returns:
//   the value (string) in the requested format
// *********************************

// implemented also in .js library for UL
string sdFormatValue(
  float f,                     // The value you want to format
  string format                // a format like'+99.999'
)
{
  string result;
  string sprintfFormat;
  bool showSign = false;
  bool leading0;
  int width;
  int digits;
  int totalLen;

  //Formatting examples:
  //  "+099.99" --> value 1.234 is shown as "+001.23"
  //  "099.99"  --> value 1.234 is shown as "001.23"
  //  "99.99"   --> value 1.234 is shown as "1.23"
  
  //First check if a '+' sign must be shown, this is requested 
  //if the first character is a '+', a 's' or a 'S'
  //
  //We also check if leading zero's must be shown
  // (if a sign is requested, then the leading zero is requested
  //  by the second character, otherwise in the first character)
  if( (format[0] == "+") || (format[0] == "s") || (format[0] == "S"))
  {
    showSign = true;
    leading0 = ( format[1] == "0" );     
  }
  else
  {
    leading0 = ( format[0] == "0" );     
  }

  totalLen = strlen( format );
  
  // Determine the width, if a sign was requested, then subract that from the lenght
  width = strlen( format ) - (int)showSign;
  
  // Now determine the digits, search for the "."
  int pos = strpos( format, ".");
  
  //if a point is found, thet determine the number of digits
  //else the digits are 0
  if( pos > 0 )
  {
    digits = totalLen - pos - 1;
  }
  else
  {
    digits = 0;
  }
  
  //Now make the format string
  if( leading0 )
  {
    //With leading zero
    sprintfFormat = "%0" + width + "." + digits + "f";
  }
  else
  {
    //Without leading zero
    sprintfFormat = "%" + width + "." + digits + "f";
  }

  // Now produce the actual value  
  sprintf( result, sprintfFormat, f );
  
  // Add the '+' sign if requested, and the value is positive
  if( showSign &&  (f >= 0.0) )
  {
    result = "+" + result;
  }
  
  return result;
}


// *********************************
// sdRotate(). Rotate a shape
//
// Returns:
//   the value (string) in the requested format
// *********************************

// implemented also in .js library for UL
void sdRotate(
  string shapeName,       // the shape that we rotate
  float value,            // value that we use to calculate the angle
  float minRange,         // Minimum range
  float maxRange,         // Maximum range
  int   startAngle,       // Starting angle 
  int   endAngle,         // Ending angle
  int   direction,         // 0 = clockwise, 1 = counter clockwise
  bool bUsePvRange = false// pvrange used
)
{  
  float range;
  
  // Use 'value', 'minRange' and 'maxRange' to determine the 
  // fraction of the angle
  float fraction = (value - minRange) / (maxRange - minRange );

  
  // Make sure the fraction is not out of range
  if( fraction > 1.0 ) fraction = 1.0;
  if( fraction < 0.0 ) fraction = 0.0;
  
  // Clockwise
  if (direction == 0)
  {
     
    range = (startAngle - endAngle);
  }
  // Counter clockwise
  else
  {
    range = (endAngle - startAngle) ;
  }
    
  if (range < 0)                   // Always turn the range into 
    range = range * -1.0;          // BT 2973 dschneider, IM 117625
    
  
  if( direction == 0 )             // clockwise
    range = range * -1.0;          // so we need to subtract angle
    
  // Now calculate the actual angle
  float angle = (float)startAngle + (fraction * (float)range);

  // Nowe actually rotate the shape
  setValue( shapeName, "rotation", angle );
}  


// *********************************
// sdVisibility(). Hide/shown a given set of shapes
//
// Returns:
//   nothing
// *********************************

// implemented also in .js library for UL
void sdVisibility(
  bool rule,
  int numShapes,
  ...  
)
{
  va_list parameters;

  int len = va_start(parameters);  // Returns the number of parameters

  for(int i = 1; i <= len; i++)
  {
    string shapeName = va_arg(parameters);


    setValue( shapeName, "visible", rule );
  }  
  
  va_end( parameters );

}

// *********************************
// sdEnableDisable(). Hide/shown a given set of shapes
//
// Returns:
//   nothing
// *********************************

// implemented also in .js library for UL
void sdEnableDisable(
  bool rule,
  int numShapes,
  ...  
)
{
  va_list parameters;

  int len = va_start(parameters);  // Returns the number of parameters

  for (int i = 1; i <= len; i++)
  {
    string shapeName = va_arg(parameters);

    setValue(shapeName, "enabled", rule);
  }  
  
  va_end( parameters );

}

// ******************************
// Name : sdAssignColor
//
// This method is used to set the color for 1 to N shapes.
// 
// Arguments:
//    numShapes       Number of shapes
//    1..N            Name of shape followed by foreground,background
//    numRules        How many rules
//    1..N            A rule followed by a color
//
// *******************************

// implemented also in .js library for UL
void sdAssignColor(
  int numShapes,
  ...
)
{
  dyn_string shapeNames;        // Name of the shape
  dyn_string shapeModes;        // The 'mode' what attribute to set ( 1 = foreground, 2 = background, 3 = both )
  bool ruleExecuted = false;
  
  // We are going to do variable arguments here
  va_list parameters;

  // How many parameters are there 
  int len = va_start(parameters);  // Returns the number of parameters

  // Lets firs read shape and color 
  for( int t = 1; t <= numShapes; t++)
  {
    string shapeName = (string)va_arg(parameters);
    int    shapeMode = (int)va_arg(parameters);
    
    // Remember the name of the shape and its mode
    dynAppend(  shapeNames, shapeName );
    dynAppend(  shapeModes, shapeMode );
  }  
  
  // The next argument will be number of rules
  int numRules = (int)va_arg(parameters);
  
 
  for( int t = 1; (t <= numRules) && !ruleExecuted; t++)
  {
    // Read the boolean value of the rule and the associated color
    bool ruleValue = (bool)va_arg(parameters);
    string color   = (string)va_arg(parameters);
    
   
    // When the rule evaluates to true
    // then assign the color
    if( ruleValue )
    {

      // We've found a rule that matches !    
      ruleExecuted = true;
      
      for( int i = 1; i <= numShapes; i++)
      {
        switch( shapeModes[i] )
        {
          case 1 : setValue( shapeNames[i], "foreCol", color ); break;
          case 2 : setValue( shapeNames[i], "backCol", color ); break;
          case 3 : setValue( shapeNames[i], "color"  , color ); break;
        }
      }      
      
    }
  }

 
  va_end( parameters );
  
  
}

// implemented also in .js library for UL
void sdShowBadConnect( string shapeName, int mode)
{
  switch( mode )
  {
    case 1 : setValue( shapeName, "foreCol", "_dpdoesnotexist" ); break;
    case 2 : setValue( shapeName, "backCol", "_dpdoesnotexist" ); break;
    case 3 : setValue( shapeName, "color"  , "_dpdoesnotexist" ); break;
      
    // When the mode is wrong, then we'll just take the entire symbol
    // ( should not happen. Only when they do something manually )  
    default:
      setValue( shapeName, "color"  , "_dpdoesnotexist" ); break;
      
  }
}

// implemented also in .js library for UL
void sdShowNoUserPermission(string shapeName, int mode)
{
  switch( mode )
  {
    case 1 : setValue( shapeName, "foreCol", "black" ); break;
    case 2 : setValue( shapeName, "backCol", "black" ); break;
    case 3 : setValue( shapeName, "color"  , "black" ); break;
    case 4 : setValue( shapeName, "enabled"  , false ); break;
      
    // When the mode is wrong, then we'll just take the entire symbol
    // ( should not happen. Only when they do something manually )  
    default:
      setValue( shapeName, "color"  , "black" ); break;
      
  }
}

// implemented also in .js library for UL
void sdResetShowNoUserPermission( 
  string shapeName,      // The name of the shape or which you want to change the color
  int mode
)
{
  switch( mode )
  {
    case 1 : setValue( shapeName, "foreCol", "" ); break;
    case 2 : setValue( shapeName, "backCol", "" ); break;
    case 3 : setValue( shapeName, "color"  , "" ); break;
    case 4 : setValue( shapeName, "enabled"  , true ); break;
      
    // When the mode is wrong, then we'll just take the entire symbol
    // ( should not happen. Only when they do something manually )  
    default:
      setValue( shapeName, "color"  , "" ); break;      
  }
}

// ******************************
// Name : sdShowBitmap
//
// This method is used to fit a bitmap in a shape
// 
// Arguments:
//    Name of shape 
//    numRules        How many rules
//    1..N            A rule followed by a bitmap
//
// *******************************

// implemented also in .js library for UL
void sdShowBitmap(
  string shapeName,
  ...
)
{
  bool ruleExecuted = false;
  
  // We are going to do variable arguments here
  va_list parameters;

  // How many parameters are there 
  int len = va_start(parameters);  // Returns the number of parameters

  // The next argument will be number of rules
  int numRules = (int)va_arg(parameters);
  
 
  for( int t = 1; (t <= numRules) && !ruleExecuted; t++)
  {
    // Read the boolean value of the rule and the associated color
    bool ruleValue  = (bool)va_arg(parameters);
    string bitmap   = (string)va_arg(parameters);
    
    // When the rule evaluates to true
    // then assign the color
    if( ruleValue )
    {

      // We've found a rule that matches !    
      ruleExecuted = true;
      
      string bitmapFile = bitmap;
      setValue( shapeName, "fill", "[pattern,[fit,any," + bitmapFile + "]]" );
  
    }
  }

  va_end( parameters );  
  
}


global bool typeMappingInitialized = false;
global mapping symbolTypeMapping;
global mapping faceplateTypeMapping;

// UL calls this at parse time
//obsolete IM 107056
string sdGetDollarParameter(string dp, string dollarName, string dollarValue)
{  
  if ( !typeMappingInitialized )
    sdInitTypeMapping();

  if ( strlen(dollarValue) != 0 ) // if the given $-Parameter in the symbol is not empty use it
    return dollarValue;
  
  if (dp == "") return "";

  //look up the configuration for dollarName
  string refFileName;
  getValue(this.refName, "panelFileName", refFileName);
  return dp + symbolTypeMapping[dpTypeName(dp)][refFileName][dollarName];
}

// implemented also in .js library for UL
string sdCheckDpe(string dp, ...)
{
  va_list parameters;
  int numParams = va_start(parameters);

  for (int i = 0; i < numParams; i++)
  {
    if ( strlen((string)va_arg(parameters)) != 0 )    
    {
      va_end(parameters);
      return dpExists(dp) ? dp : "";
    }
  }
  
  va_end(parameters);
  
  return dpExists(dp) ? dp : "";
}

// UL calls this at parse time
int sdGetFaceplateFeatures(string dp)
{
  return faceplateTypeMapping[dpTypeName(dp)]["faceplateDefaultFeatures"];
}

// UL calls this at parse time
string sdGetFaceplateMain(string dp)
{
  return (mappingHasKey(faceplateTypeMapping, dpTypeName(dp)) && 
          mappingHasKey(faceplateTypeMapping[dpTypeName(dp)], "faceplateMain")) ? 
            faceplateTypeMapping[dpTypeName(dp)]["faceplateMain"] : 
            "";
}

// UL calls this at parse time
dyn_string sdGetFaceplateUserDefTabPanels(string dp)
{
  return faceplateTypeMapping[dpTypeName(dp)]["faceplateUserDefTabPanels"];
}

// UL calls this at parse time
dyn_string sdGetFaceplateUserDefTabNames(string dp)
{
  return faceplateTypeMapping[dpTypeName(dp)]["faceplateUserDefTabNames"];
}

// UL calls this at parse time
dyn_string sdGetFaceplateUserDefDollarParams(string dp)
{
  return faceplateTypeMapping[dpTypeName(dp)]["dollarParams"];
}

// UL calls this at parse time
dyn_string sdGetChangingValues(string dp)
{
  return faceplateTypeMapping[dpTypeName(dp)]["changingValues"];
}

// UL calls this at parse time
dyn_string sdGetDisplayValues(string dp)
{
  return faceplateTypeMapping[dpTypeName(dp)]["displayValues"];
}

// UL calls this at parse time
dyn_string sdGetPeripheryAddress(string dp)
{
  return faceplateTypeMapping[dpTypeName(dp)]["peripheryAddress"];
}

// UL calls this at parse time
string sdGetNoteDPE(string dp)
{
  return faceplateTypeMapping[dpTypeName(dp)]["noteDPE"];
}

// Creates a list of DPs out of the dpname and its elements.
// Returns only existing DPs (one can call dpGet, dpConnect, ... on)
// @dp The dpname ($DP)
// @dpes The dp elements (".x.y.z")
dyn_string sdGetValidDps(string dp, dyn_string dpes)
{
  dyn_string dps = makeDynString();
  for (int i = 1; i <= dynlen(dpes); i++)
  {
    string checkableDp = dpSubStr(dp + dpes[i], DPSUB_DP_EL);
    if (dpExists(checkableDp + ":_online.._value")) dynAppend(dps, checkableDp); 
  }
  return dps;
}

private synchronized void sdInitTypeMapping()
{
  if ( typeMappingInitialized ) // already initialized
    return;
    
  dyn_string dps = dpNames("*", "_objWizardSymbolsPerType");
  
  for (int i = 1; i <= dynlen(dps); i++)
    sdReInitTypeMapping(dps[i]);
  
  dpQueryConnectSingle("sdTypeMappingCB", false, "", "SELECT '_online.._value' FROM '*' WHERE _DPT = \"_objWizardSymbolsPerType\"");
  dpQueryConnectSingle("sdSymbolConfigurationCB", false, "", "SELECT '_online.._value' FROM '*' WHERE _DPT = \"_objWizardSymbolConfiguration\"");
  
  typeMappingInitialized = true;
}

mapping symbolToDPTHelper;
mapping symbolConfHelper;
private void sdReInitTypeMapping(string dp)
{
  string dptName;
  dyn_string symbolList;
  string faceplateMain;
  bit32 faceplateDefaultFeatures;
  dyn_string faceplateUserDefTabNames;
  dyn_string faceplateUserDefTabPanels;
  dyn_string faceplateUserDefTabDollarParams;
  dyn_string displayValues, changingValues, peripheryAddress;
  string noteDPE;
  dpGet(dp + ".DPTName", dptName,
        dp + ".symbolTypeMapping", symbolList,
        dp + ".faceplateMain", faceplateMain,
        dp + ".faceplateDefaultFeatures", faceplateDefaultFeatures,
        dp + ".faceplateUserDefTabPanels", faceplateUserDefTabPanels, 
        dp + ".faceplateUserDefTabNames", faceplateUserDefTabNames, 
        dp + ".defaultFeatureConfig.displayValues", displayValues, 
        dp + ".defaultFeatureConfig.changingValues", changingValues, 
        dp + ".defaultFeatureConfig.peripheryAddress", peripheryAddress, 
        dp + ".defaultFeatureConfig.noteDPE", noteDPE,
        dp + ".faceplateUserDefTabDollarParams", faceplateUserDefTabDollarParams
        );
  
  symbolConfHelper[dp] = makeDynAnytype(dptName, symbolList);

  mapping m;
  m["faceplateMain"] = faceplateMain;
  m["faceplateDefaultFeatures"] = faceplateDefaultFeatures;
  m["faceplateUserDefTabNames"] = faceplateUserDefTabNames;
  m["faceplateUserDefTabPanels"] = faceplateUserDefTabPanels;
  m["displayValues"] = displayValues;
  m["changingValues"] = changingValues;
  m["peripheryAddress"] = peripheryAddress;
  m["noteDPE"] = noteDPE;
  m["dollarParams"] = faceplateUserDefTabDollarParams;
  faceplateTypeMapping[dptName] = m;

  mapping symbols;
  for (int j = 1; j <= dynlen(symbolList); j++)
  {    
    string fileName;
    dyn_string dollarParameters;
    dyn_string dollarValues;
    dpGet(symbolList[j] + ".symbolFileName", fileName,
          symbolList[j] + ".dollarParameters", dollarParameters,
          symbolList[j] + ".dollarValues", dollarValues);

    symbolToDPTHelper[dpSubStr(symbolList[j], DPSUB_DP)] = makeDynAnytype(dptName, fileName, dollarParameters, dollarValues);

    mapping dollars;
    for (int k = 1; k <= dynlen(dollarParameters); k++)
      dollars[dollarParameters[k]] = dollarValues[k];
      
    symbols[fileName] = dollars;
  }

  symbolTypeMapping[dptName] = symbols;
}

private void sdTypeMappingCB(string userData, dyn_dyn_anytype values)
{
  for (int i = 2; i <= dynlen(values); i++)
  {
    string dp = dpSubStr(values[i][1], DPSUB_DP);
    
    if ( !mappingHasKey(symbolConfHelper, dp) )
    {
      sdReInitTypeMapping(dp);
      continue;
    }

    string dpt = symbolConfHelper[dp][1];
    string dpId = (string)values[i][1];
    string dpEl = substr(dpId, strpos(dpId, ".") + 1);

    switch ( dpEl )
    {
      case "DPTName":
      {
        mapping tmp = faceplateTypeMapping[dpt];
        mappingRemove(faceplateTypeMapping, dpt);
        faceplateTypeMapping[values[2][1]] = tmp;
        
        tmp = symbolTypeMapping[dpt];
        mappingRemove(symbolTypeMapping, dpt);
        symbolTypeMapping[values[2][1]] = tmp;
        
        symbolConfHelper[dp] = values[2][1];
        
        dyn_string symbolDps = symbolConfHelper[dp][2];
        for (int j = 1; j <= dynlen(symbolDps); j++)
        {
          dyn_anytype da = symbolToDPTHelper[symbolDps[j]];
          da[1] = values[2][1];
          symbolToDPTHelper[symbolDps[j]] = da;
        }
        break;
      }
      
      case "symbolTypeMapping":
        sdReInitTypeMapping(dpSubStr(values[2][1], DPSUB_DP));
      break;

      default:
        faceplateTypeMapping[dpt][dpEl] = values[i][2];
      break;
    }
  }
}

private void sdSymbolConfigurationCB(string userData, dyn_dyn_anytype values)
{
  for (int i = 2; i <= dynlen(values); i++)
  {
    string dp = dpSubStr(values[i][1], DPSUB_DP);
    string dpt = symbolToDPTHelper[dp][1];
    string fileName = symbolToDPTHelper[dp][2];
    string dpId = (string)values[i][1];
    string dpEl = substr(dpId, strpos(dpId, ".") + 1);
    
    switch ( dpEl )
    {
      case "symbolFileName":
      {
        mapping tmp = symbolTypeMapping[dpt][fileName];
        mappingRemove(symbolTypeMapping[dpt], fileName);
        symbolTypeMapping[dpt][values[i][2]] = tmp;
        
        symbolToDPTHelper[dp] = makeDynString(dpt, values[i][2]);
        break;
      }

      case "dollarParameters":
      {
        dyn_string dollarValues = symbolToDPTHelper[dp][4];
        
        mapping dollars;
        for (int k = 1; k <= dynlen(dollarParameters); k++)
          dollars[values[i][2][k]] = dollarValues[k];
        
        symbolTypeMapping[dpt][fileName] = dollars;
        break;
      }

      case "dollarValues":
      {
        dyn_string dollarParameters = symbolToDPTHelper[dp][4];
        
        mapping dollars;
        for (int k = 1; k <= dynlen(dollarParameters); k++)
          dollars[dollarParameters[k]] = values[i][2][k];
        
        symbolTypeMapping[dpt][fileName] = dollars;
        break;
      }
    }
  }
}

// implemented also in .js library for UL
void sdOpenFaceplate(string Dp, string symbolFileName, int x = -1, int y = -1, string title)
{
  if ( !typeMappingInitialized )
    sdInitTypeMapping();
  
  bool openEnabled = false;

  if (dpExists(Dp))
  {
    string typeName = dpTypeName(Dp);
    if ((strlen(typeName) > 0) && (mappingHasKey(faceplateTypeMapping, typeName)))
      openEnabled = true;
  }

  if (openEnabled)
  {
    string s;
    dyn_string dollars = makeDynString("$DP:" + Dp, "$SYMBOL:" + symbolFileName, "$STRING_Title:" + title);
    
       
    if ( (x != -1) && (y != -1) )
      ChildPanelOnModal("gedi/Symple/SympleFaceplate.xml", Dp + "_faceplate", dollars, x, y);
    else
      ChildPanelOnCentralModal("gedi/Symple/SympleFaceplate.xml", Dp + "_faceplate", dollars);

  }
}

void sdDpGet(dyn_mapping& params, ...)
{
  for (int i = 1; i <= dynlen(params); ++i)
  {
    anytype var;
    
    string dp = params[i]["DP"];
    int type = params[i]["TYPE"];

    if (type == SD_DPGETTYPE_DP_GET)
    {
      dpGet(dp, var);
      params[i]["VALUE"] = var;
    }
    else if (type == SD_DPGETTYPE_DP_GET_DESCRIPTION)
    {
      params[i]["VALUE"] = dpGetDescription(dp);
    }
    else if (type == SD_DPGETTYPE_DP_GET_COMMENT)
    {
      params[i]["VALUE"] = dpGetComment(dp);
    }
    else if (type == SD_DPGETTYPE_DP_GET_ALIAS)
    {
      params[i]["VALUE"] = dpGetAlias(dp);
    }
    else if (type == SD_DPGETTYPE_DP_GET_UNIT)
    {
      params[i]["VALUE"] = dpGetUnit(dp);
    }
    else
    {
      params[i]["VALUE"] = "";
    }
  }
}

int sdDpSet(int permission, ...)
{
  va_list parameters;
  dyn_string dpnames = makeDynString();
  dyn_anytype values = makeDynString();

  if (getUserPermission(permission))
  {
    int len = va_start(parameters); // Returns the number of parameters
    if (len > 0)
    {
      int j = 1;
      for(int i = 1; i <= len; i += 2)
      {
        dpnames[j] = va_arg(parameters);
        values[j] = va_arg(parameters);
        j++;    //IM 109907 don't use dynAppend as it takes elements of dyns as singletons
      }  
      va_end( parameters );
    
      return dpSetWait(dpnames, values);
    }
  }
  
  return -1;
}

string sdToggleBit(string Dp)
{
  bool value;
  dpGet(Dp, value);
  value = !value;
  dpSet(Dp, value);
  return "";
}

void sdCalculate(int permission, string code, anytype& variable)
{
  if (getUserPermission(permission))
    variable = code;
}

void sdGetUserPermission(int permission, anytype variable)
{
  // ignore
}

void sdNavigate(int type, int ptSelect, int histSelect, string panelName, string sModuleName, ...)
{
  if (type == 1)
  {
    // Navigate through PT
    if (ptSelect == 0) // Forward
      pt_panelOn(panelName, sModuleName, 2);
    else if (ptSelect == 1) // Back
      pt_panelOn(panelName, sModuleName, 1);
    else if (ptSelect == 2) // Up
      pt_panelOn(panelName, sModuleName, 3);
    else if (ptSelect == 3) // Down
      pt_panelOn(panelName, sModuleName, 4);
  }
  else if (type == 2)
  {
    // Navigate through panel history
    if (histSelect == 0) ptnavi_HistoryForward();
    else if (histSelect == 1) ptnavi_HistoryBack();
  }
  else if (type == 3)
  {
    // Load a panel
    va_list parameters;
    dyn_string args = makeDynString();
    int len = va_start(parameters);
    if (len > 1)   // IM 109596
    {
      for(int i = 1; i <= len; i += 2)
      {
        string dollar = va_arg(parameters);
        if (dollar[0] != '$') dollar = "$" + dollar; // if $parameter has no leading $ insert it
        dynAppend(args, dollar + ":" + va_arg(parameters));
      }
      va_end( parameters );
    }
    else if ( len == 1 )
    {
      args = va_arg(parameters);
    }
    string fullPanelName = getPath(PANELS_REL_PATH) + panelName;
    RootPanelOnModule(fullPanelName, "", sModuleName, args);
  }
}

void sdUserDefinedFunction(string functionName, ...) //TODO
{
  //do nothing
}

void sdGetCatStr(string cat, string key, string s)
{
  // ignore
}

bool isUltralight() 
{ 
  if (isFunctionDefined("getApplicationProperty") && getApplicationProperty("platformName") == "ulc") 
    return true; 
  else 
    return false; 
} 

int sdDpCopy(string src, string dest, int iDriver = -1)
{
  int iErr;
  dpCopy( src, dest, iErr, iDriver);
  return iErr;
}

//IM 109106: dummy function for ulc
time sdConvertToTime(anytype aTime)
{ 
  return (time)aTime;
}

/** 
  @author mschmit
  Function display a context menu
  @param iEntries ... number of entries
  @param var list - int iType, int iParentId, int iID, string sName, string sEnableRule
  @return returns the selected ID. If no selectet return 0 
*/
int sdOpenContextMenu(int iEntries, ...)
{
  va_list parameters;
  dyn_anytype daParameters;

  int len = va_start(parameters);  // Returns the number of parameters

  for(int i = 1; i <= len; i++)
  {
    anytype aValue = va_arg(parameters);
    dynAppend(daParameters, aValue);
  }    
  va_end( parameters );

  dyn_dyn_anytype ddaMenu;
  
  for(int i = 1; i < dynlen(daParameters); i+=6)
    dynAppend(ddaMenu, makeDynAnytype(daParameters[i],daParameters[i+1],daParameters[i+2],daParameters[i+3],daParameters[i+4]));

  dynDynSort(ddaMenu, 2);

  dyn_string dsMenu;
  int iAnswer;
  mapping mCascade;
  
  for(int i = 1; i <= dynlen(ddaMenu); i++)//builds menu array
  {
    int iType = ddaMenu[i][1];
    int iParentId = ddaMenu[i][2];
    int iId = ddaMenu[i][3];
    string sText = ddaMenu[i][4];
    int bEnabled = ddaMenu[i][5]; 
    
    if ( iType == 1 )//Button
    {
      if (mappingHasKey(mCascade, iParentId))
        if (!dynContains(dsMenu, mCascade[iParentId]))
          dynAppend(dsMenu, mCascade[iParentId]);
      
      dynAppend(dsMenu, makeDynString("PUSH_BUTTON, "+sText+", "+(string)iId+", "+(string)bEnabled+""));
    }
    else if ( iType == 2 )//Separator
    {
      if (mappingHasKey(mCascade, iParentId))
        if (!dynContains(dsMenu, mCascade[iParentId]))
          dynAppend(dsMenu, mCascade[iParentId]);
            
      dynAppend(dsMenu, makeDynString("SEPARATOR"));
    }
    else if ( iType == 3 )//Cascade
    {
      if (mappingHasKey(mCascade, iParentId))
        if (!dynContains(dsMenu, mCascade[iParentId]))
          dynAppend(dsMenu, mCascade[iParentId]);
            
      dynAppend(dsMenu, makeDynString("CASCADE_BUTTON, "+sText+", "+(string)bEnabled+""));
      mCascade[iId] = sText;
    }
  }  
  popupMenu(dsMenu, iAnswer);  
  return iAnswer;
}

void sdLogout(bool question,  bool erasePanel, bool quitWindow)
{
  int closeModules = 0; //default "opens new login module, doesn't close other modules"
  if(erasePanel && quitWindow) // "opens new login module, closes other modules"
  {
    closeModules = 1;
  }
  else if(!erasePanel && !quitWindow) //"doesn't open new login module, doesn't close other modules"
  {
    closeModules = 2;
  }
  else if(!erasePanel && quitWindow) //"doesn't open new login module, closes other modules"
  {
    closeModules = 3;
  }
  LogoutDialog(question, closeModules, "", "");
}

dyn_int sdGetPosition(string sShapeName)
{
  int x, y;

  getValue(sShapeName, "position", x, y);
  
  return makeDynInt(x, y);
}

int sdGetLastError()
{
  if ( dynlen(getLastError()) > 0 )
    return -1;
  else
    return 0;
  
}

void sdErrorDialog(anytype error)
{
  errorDialog(error);
}
