#include <QmlDp.hxx>

#include <Manager.hxx>
#include <AttributeNrs.hxx>
#include <ErrHdl.hxx>
#include <QtUtils.hxx>

#include <DynVar.hxx>
#include <AnyTypeVar.hxx>
#include <TextVar.hxx>
#include <FloatVar.hxx>
#include <BitVar.hxx>
#include <IntegerVar.hxx>
#include <UIntegerVar.hxx>
#include <TimeVar.hxx>
#include <CharVar.hxx>
#include <LongVar.hxx>
#include <ULongVar.hxx>
#include <Bit32Var.hxx>
#include <Bit64Var.hxx>
#include <MappingVar.hxx>

#include <QCursor>
#include <QPoint>
#include <QPointF>
#include <QSize>
#include <QSizeF>
#include <QRect>
#include <QRectF>
#include <QPolygon>
#include <QBitArray>

//--------------------------------------------------------------------------------
//--------------------------------------------------------------------------------

void QmlHL::hotLinkCallBack(DpMsgAnswer &answer)
{
  AnswerGroup *group = answer.cutFirstGroup();
  DpHLGroup hlGroup;
  hlGroup.swallowAnswer(*group);
  delete group;
  hotLinkCallBack(hlGroup);
}

//--------------------------------------------------------------------------------

void QmlHL::hotLinkCallBack(DpHLGroup &group)
{
  DpVCItem *item = group.getFirstItem();
  Variable *value;
  if ( item && (value = item->getValuePtr()) )
  {
    qmlDp->emitNewValue(QmlDp::VariableToQVariant(value));
  }
}

//--------------------------------------------------------------------------------
//--------------------------------------------------------------------------------

QmlDp::QmlDp(QObject *parent)
  : QObject(parent), varType(NO_VAR), hl(0)
{
}

//--------------------------------------------------------------------------------

QmlDp::~QmlDp()
{
  if ( hl )
    Manager::dpDisconnect(dpId, hl);

  delete hl;
}

//--------------------------------------------------------------------------------

void QmlDp::componentComplete()
{
  CharString dpStr = QtUtils::actLangEncoded(dpa);

  if ( !Manager::getId(dpStr, dpId) || dpId.isNull() )
  {
    ErrHdl::error(ErrClass::PRIO_WARNING, ErrClass::ERR_PARAM, ErrClass::DP_NOT_EXISTENT, dpStr);
    return;
  }

  // if no "_online.._value" given, use it by default for dpConnect
  if ( dpId.getConfig() == DPCONFIGNR_NOCONFIGNR )
  {
    dpId.setConfig(DPCONFIGNR_ONLINEVALUE);   // _online
    dpId.setAttr(DPVALUE_VALUE_ATTR);         // _value
  }

  Manager::getDpIdentificationPtr()->getAttributeType(dpId, varType);

  Manager::dpConnect(dpId, hl = new QmlHL(this), PVSS_FALSE);
}

//--------------------------------------------------------------------------------

void QmlDp::setValue(const QVariant &newValue)
{
  if ( dpId.isNull() )
    return;

  DpIdentifier dp(dpId);

  // if no "_original.._value" given, use it by default
  if ( (dp.getConfig() == DPCONFIGNR_NOCONFIGNR) ||
       (dp.getConfig() == DPCONFIGNR_ONLINEVALUE) )  // or _online
  {
    dp.setConfig(DPCONFIGNR_DPVALUE);   // _original
    dp.setAttr(DPVALUE_VALUE_ATTR);     // _value
  }

  Variable *givenVar = QVariantToVariable(newValue);
  Variable *neededVar = Variable::allocate(varType);

  if ( givenVar && neededVar )
  {
    *neededVar = *givenVar;
    Manager::dpSet(dp, *neededVar);
  }

  delete givenVar;
  delete neededVar;
}

//--------------------------------------------------------------------------------

void QmlDp::emitNewValue(const QVariant &newValue)
{
  if ( newValue == value )
    return;

  value = newValue;

  emit valueChanged(value);
}

//--------------------------------------------------------------------------------

Variable *QmlDp::QVariantToVariable(const QVariant &var)
{
  // Qt5 doc:
  // Although this function is declared as returning QVariant::Type(obsolete),
  // the return value should be interpreted as QMetaType::Type
  switch ( static_cast<QMetaType::Type>(var.type()) )
  {
    case QMetaType::UnknownType: return new AnyTypeVar();

    case QMetaType::Bool      : return new BitVar     (var.toBool());
    case QMetaType::Double    : return new FloatVar   (var.toDouble());
    case QMetaType::Float     : return new FloatVar   (var.toFloat());

    case QMetaType::LongLong  : return new LongVar    (var.toLongLong());

    case QMetaType::Short     :
    case QMetaType::Int       :
    case QMetaType::Long      : return new IntegerVar (var.toInt());

    case QMetaType::ULongLong : return new ULongVar   (var.toULongLong());

    case QMetaType::UShort    :
    case QMetaType::UInt      :
    case QMetaType::ULong     : return new UIntegerVar(var.toUInt());

    case QMetaType::QByteArray: return new TextVar    (var.toByteArray());
    case QMetaType::QCursor   : return new IntegerVar (var.value<QCursor>().shape());
    case QMetaType::QUrl      : return new TextVar    (var.toUrl().toEncoded());

    case QMetaType::QString:
    {
      return new TextVar(QtUtils::actLangEncoded(var.toString()));
    }

    case QMetaType::QDate:
    {
      QDateTime date(var.toDate());
      if ( date.isValid() )
        return new TimeVar(PVSSTime(date.toTime_t(), 0));
      else
        return new TimeVar(0, 0);
    }

    case QMetaType::QTime:
    {
      QTime t(var.toTime());
      return new TimeVar(PVSSTime(QTime(0, 0).secsTo(t), t.msec()));
    }

    case QMetaType::QDateTime:
    {
      QDateTime t(var.toDateTime());
      return new TimeVar(PVSSTime(t.toTime_t(), t.time().msec()));
    }

    case QMetaType::QStringList:
    {
      DynVar *dyn = new DynVar(TEXT_VAR);
      QStringList list = var.toStringList();

      for (int i = 0; i < list.count(); i++)
        dyn->append(new TextVar(QtUtils::actLangEncoded(list[i])));

      return dyn;
    }

    case QMetaType::QVariantList:
    {
      DynVar *dyn = new DynVar(ANYTYPE_VAR);
      QVariantList list = var.toList();

      for (int i = 0; i < list.count(); i++)
        dyn->append(new AnyTypeVar(QVariantToVariable(list[i])));

      return dyn;
    }

    case QMetaType::QPoint:
    {
      DynVar *dyn = new DynVar(INTEGER_VAR);

      QPoint p = var.toPoint();

      dyn->append(new IntegerVar(p.x()));
      dyn->append(new IntegerVar(p.y()));

      return dyn;
    }

    case QMetaType::QPointF:
    {
      DynVar *dyn = new DynVar(FLOAT_VAR);

      QPointF p = var.toPointF();

      dyn->append(new FloatVar(p.x()));
      dyn->append(new FloatVar(p.y()));

      return dyn;
    }

    case QMetaType::QSize:
    {
      DynVar *dyn = new DynVar(INTEGER_VAR);

      QSize s = var.toSize();

      dyn->append(new IntegerVar(s.width()));
      dyn->append(new IntegerVar(s.height()));

      return dyn;
    }

    case QMetaType::QSizeF:
    {
      DynVar *dyn = new DynVar(FLOAT_VAR);

      QSizeF s = var.toSizeF();

      dyn->append(new FloatVar(s.width()));
      dyn->append(new FloatVar(s.height()));

      return dyn;
    }

    case QMetaType::QRect:
    {
      DynVar *dyn = new DynVar(INTEGER_VAR);

      QRect r = var.toRect();

      dyn->append(new IntegerVar(r.x()));
      dyn->append(new IntegerVar(r.y()));
      dyn->append(new IntegerVar(r.width()));
      dyn->append(new IntegerVar(r.height()));

      return dyn;
    }

    case QMetaType::QRectF:
    {
      DynVar *dyn = new DynVar(FLOAT_VAR);

      QRectF r = var.toRectF();

      dyn->append(new FloatVar(r.x()));
      dyn->append(new FloatVar(r.y()));
      dyn->append(new FloatVar(r.width()));
      dyn->append(new FloatVar(r.height()));

      return dyn;
    }

    case QMetaType::QPolygon:
    {
      DynVar *dyn = new DynVar(DYNINTEGER_VAR);
      QPolygon points = var.value<QPolygon>();

      for (int i = 0; i < points.count(); i++)
      {
        DynVar *point = new DynVar(INTEGER_VAR);
        point->append(new IntegerVar(points[i].x()));
        point->append(new IntegerVar(points[i].y()));
        dyn->append(point);
      }

      return dyn;
    }

    case QMetaType::QBitArray:
    {
      QBitArray barr = var.toBitArray();

      if ( barr.count() <= 32 )
      {
        Bit32Var *bvar = new Bit32Var;
        for (int i = 0; i < barr.count(); i++)
          bvar->setBit(i, barr.at(i));

        return bvar;
      }
      else if ( barr.count() <= 64 )
      {
        Bit64Var *bvar = new Bit64Var;
        for (int i = 0; i < barr.count(); i++)
          bvar->setBit(i, barr.at(i));

        return bvar;
      }
      else
        return 0;
    }

    case QMetaType::QVariantMap:
    {
      MappingVar *map = new MappingVar;

      QMap<QString, QVariant> qmap = var.toMap();
      QMap<QString, QVariant>::const_iterator it;

      for (it = qmap.constBegin(); it != qmap.constEnd(); ++it)
      {
        map->setAt(new TextVar(QtUtils::actLangEncoded(it.key())),
                   QVariantToVariable(it.value()));
      }

      return map;
    }

    default:
      return 0;
  }
}

//---------------------------------------------------------------------------

QVariant QmlDp::VariableToQVariant(const Variable *var)
{
  if ( !var )
    return QVariant();

  switch ( var->isA() )
  {
    case BIT_VAR     : return QVariant(static_cast<bool>(static_cast<const BitVar*>(var)->getValue()));
    case FLOAT_VAR   : return QVariant(static_cast<const FloatVar*>(var)->getValue());
    case UINTEGER_VAR: return QVariant(static_cast<unsigned int>(static_cast<const UIntegerVar*>(var)->getValue()));

    case INTEGER_VAR :
      return QVariant(static_cast<int>(static_cast<const IntegerVar*>(var)->getValue()));

    case LONG_VAR : return QVariant(static_cast<long long>(static_cast<const LongVar*>(var)->getValue()));
    case ULONG_VAR: return QVariant(static_cast<unsigned long long>(static_cast<const ULongVar*>(var)->getValue()));

    case TEXT_VAR:
      return QVariant(QtUtils::actLangDecoded(static_cast<const TextVar*>(var)->getString()));

    case TIME_VAR:
    {
      BC_CTime t = static_cast<const TimeVar*>(var)->getTime();
      QDate date(t.Year(), t.Month(), t.Day());
      QTime time(t.Hour(), t.Minute(), t.Second(), static_cast<const TimeVar*>(var)->getMilli());
      return QVariant(QDateTime(date, time));
    }

    case ANYTYPE_VAR:
    case MIXED_VAR:
    {
      if ( static_cast<const AnyTypeVar*>(var)->getVar() )
        return VariableToQVariant(static_cast<const AnyTypeVar*>(var)->getVar());
      else
        return QVariant();  // this is an invalid QVariant but a valid return value (failed is _not_ set)
    }

    case DYNTEXT_VAR:
    {
      QStringList list;
      const DynVar *dyn = static_cast<const DynVar*>(var);

      for (const Variable *v = dyn->getFirst(); v; v = dyn->getNext())
        list.append(QtUtils::actLangDecoded(static_cast<const TextVar*>(v)->getString()));

      return QVariant(list);
    }

    case DYNINTEGER_VAR:
    {
      const DynVar *dyn = static_cast<const DynVar*>(var);

      QList<QVariant> list;

      for (const Variable *v = dyn->getFirst(); v; v = dyn->getNext())
        list.append(VariableToQVariant(v));

      return QVariant(list);
    }

    case DYNFLOAT_VAR:
    {
      const DynVar *dyn = static_cast<const DynVar*>(var);

      QList<QVariant> list;

      for (const Variable *v = dyn->getFirst(); v; v = dyn->getNext())
        list.append(VariableToQVariant(v));

      return QVariant(list);
    }

    case MAPPING_VAR:
    {
      const MappingVar *map = static_cast<const MappingVar*>(var);
      QMap<QString, QVariant> qmap;  // aka QVariantMap

      for (unsigned int i = 0; i < map->getNumberOfItems(); i++)
        qmap.insert(QtUtils::actLangDecoded(map->getKey(i)->formatValue("")),
                    VariableToQVariant(map->getValue(i)));

      return QVariant(qmap);
    }

    case BIT32_VAR:
    {
      QBitArray barr(32);
      for (int i = 0; i < 32; i++)
        barr.setBit(i, static_cast<const Bit32Var*>(var)->getBit(i));

      return QVariant(barr);
    }

    case BIT64_VAR:
    {
      QBitArray barr(64);
      for (int i = 0; i < 64; i++)
        barr.setBit(i, static_cast<const Bit64Var*>(var)->getBit(i));

      return QVariant(barr);
    }

    default:
    {
      if ( var->isA(DYN_VAR) == DYN_VAR )
      {
        const DynVar *dyn = static_cast<const DynVar*>(var);

        QList<QVariant> list;

        for (const Variable *v = dyn->getFirst(); v; v = dyn->getNext())
          list.append(VariableToQVariant(v));

        return QVariant(list);
      }

      return QVariant();
    }
  }
}

//--------------------------------------------------------------------------------
