﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;

namespace PanelXmlToLib
{
    static class XmlToLib
    {
        #region Private helpers.
        private static void ReadEvents(XmlTextReader reader, string objectName, string objectType, ref StringBuilder script)
        {
            string name = string.Empty, value = String.Empty;
            int elementLevel = 0;

            // Loop until we reach the closing element.
            do
            {
                // Also keep track of child elements.
                if (reader.NodeType == XmlNodeType.Element) elementLevel++;
                else if (reader.NodeType == XmlNodeType.EndElement)
                {
                    if ((elementLevel--) == 1)
                    {
                        break;
                    }
                }

                // Is this the start of an element?
                if (reader.NodeType == XmlNodeType.Element)
                {
                    // Is it the "script" element?
                    if (reader.Name.ToLower() == "script")
                    {
                        // Iterate the attributes to find the "name" attribute. store
                        // the value of the attribute when found.
                        while (reader.MoveToNextAttribute())
                        {
                            if (reader.Name.ToLower() == "name")
                            {
                                name = reader.Value;
                                break;
                            }
                        }

                        // The next element should be the source code...
                        reader.Read();

                        // Replave the single newline characters by the correct
                        // line endings and append two empty lines.
                        value = System.Web.HttpUtility.HtmlDecode(reader.Value);
                        value = Environment.NewLine + Environment.NewLine + Environment.NewLine + value.Trim();


                        // We build the function name as follows:
                        //
                        // void <ObjectType>_<objectName>_<eventName>( ... )
                        //
                        // If it is the scope library we are dealing with we just copy the code verbatim.
                        if (name.ToLower() != "scopelib")
                        {
                            script.Append(Environment.NewLine + Environment.NewLine + value.Replace("main(", "void " + objectType + "_" + objectName + "_" + name + "("));
                        }
                        else
                        {
                            script.Append(Environment.NewLine + Environment.NewLine + value);
                        }
                    }
                }

                // Clear the variables.
                name = string.Empty;
                value = string.Empty;
            } while (reader.Read());
        }

        private static void ExtractPanelScripts(XmlTextReader reader, ref StringBuilder script)
        {
            int elementLevel = 0;
            string obj = "panel", type = "PANEL";

            // Loop until we reach the closing element.
            do
            {
                // Also keep track of child elements.
                if (reader.NodeType == XmlNodeType.Element) elementLevel++;
                else if (reader.NodeType == XmlNodeType.EndElement)
                {
                    if ((elementLevel--) == 1)
                    {
                        break;
                    }
                }

                // Is this the start of an element?
                if (reader.NodeType == XmlNodeType.Element)
                {
                    switch (reader.Name.ToLower())
                    {
                        case "events":
                            ReadEvents(reader, obj, type, ref script);
                            break;

                        case "shape":
                            while (reader.MoveToNextAttribute())
                            {
                                if (reader.Name.ToLower() == "shapetype")
                                {
                                    type = reader.Value;
                                }
                                else if (reader.Name.ToLower() == "name")
                                {
                                    obj = reader.Value;
                                }
                            }
                            break;
                    }
                }
            } while (reader.Read());
        }
        #endregion

        #region Public methods.
        public static void LoadXmlAndConvertToLib(string fileName, string sourcePath, string targetPath)
        {
            try
            {
                // Open the XML file.
                using (XmlTextReader reader = new XmlTextReader(fileName))
                {
                    // Buffer for the extracted source code.
                    StringBuilder script = new StringBuilder();

                    // Loop through the XML file.
                    while (reader.Read())
                    {
                        // Is it an element? If it is and it is the
                        // "panel" element we start extracting source code.
                        switch (reader.NodeType)
                        {
                            case XmlNodeType.Element:
                                switch (reader.Name.ToLower())
                                {
                                    case "panel":
                                        ExtractPanelScripts(reader, ref script);
                                        break;
                                }
                                break;
                        }
                    }

                    // Do we have a target path? If we do replace the source path from
                    // the file name with the target path leaving us with the target 
                    // filename.
                    if (!String.IsNullOrEmpty(targetPath))
                    {
                        fileName = fileName.Replace(sourcePath, targetPath);
                        
                        // Make sure the target folder exists.
                        if (Directory.Exists(Path.GetDirectoryName(fileName)) == false)
                        {
                            Directory.CreateDirectory(Path.GetDirectoryName(fileName));
                        }
                    }

                    // Open the file for writing. Overwrites any existing file!
                    using (TextWriter tw = new StreamWriter(Path.ChangeExtension(fileName, ".pnl.lib")))
                    {
                        tw.Write(script.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                // Ooops. Fuckup...
                MessageBox.Show(ex.Message, "Error...", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
        #endregion
    }
}
