#include <Ewo.hxx>

// C++
#include <algorithm>

// Qt
#include <QtGlobal>
#include <QStringBuilder>
#include <QVector>
#include <QDateTime>

#include <QPushButton>
#include <QLayout>
#include <QKeyEvent>
#include <QClipboard>

#include <QDebug>

// local
#include "DataGrid.hxx"
#include "Delegates/CheckboxDelegate.hxx"
#include "Delegates/SpinboxDelegate.hxx"
#include "Delegates/ComboboxDelegate.hxx"
#include "Delegates/ReadonlyDelegate.hxx"
#include "Delegates/ButtonDelegate.hxx"

//--------------------------------------------------------------------------------
// constructor
EWO_PLUGIN( Ewo ) DataGrid::DataGrid(QWidget *parent) : QWidget(parent), ui(new Ui::Widget), keyColumn(-1), isWriting(false)
{
	ui->setupUi(this);

	// Keep this
	QTableView *tableView = ui->tableView;
	this->model = new QStandardItemModel(this);
	proxyModel = new SortFilterProxyModel(this);
	proxyModel->setSourceModel(model);

	filter = new AndList(this);
	applyFilter();

	tableView->setModel(this->proxyModel);
	tableView->setContextMenuPolicy(Qt::CustomContextMenu);

	// default row size
	tableView->verticalHeader()->setResizeMode(QHeaderView::Fixed);
	tableView->verticalHeader()->setDefaultSectionSize(20);
	tableView->verticalHeader()->setHidden(true);

	// default column size
	tableView->horizontalHeader()->setResizeMode(QHeaderView::Interactive);

	FlowLayout* layout = new FlowLayout(ui->filterStack, 0, 0, 0);
	ui->filterStack->setLayout(layout);

	// signals to slots
	connectDataEvents();

	connect(ui->tableView, SIGNAL(customContextMenuRequested(QPoint)),SLOT(onCustomMenuRequested(QPoint)));
	connect(ui->tableView->horizontalHeader(), SIGNAL(sortIndicatorChanged(int, Qt::SortOrder)), SLOT(onSortOrderChanged(int, Qt::SortOrder)));

}

void DataGrid::connectDataEvents() {
	connect(ui->tableView->model(), SIGNAL(dataChanged(const QModelIndex&, const QModelIndex&)), this, SLOT(onDataChanged(const QModelIndex&, const QModelIndex&)));
}

void DataGrid::disconnectDataEvents() {
	disconnect(ui->tableView->model(), SIGNAL(dataChanged(const QModelIndex&, const QModelIndex&)), this, SLOT(onDataChanged(const QModelIndex&, const QModelIndex&)));
}

DataGrid::~DataGrid() {
}

// functions
bool lessThan(const QModelIndex& e1, const QModelIndex& e2) {
	if (e1.row() < e2.row()) {
		return true;
	} else if (e1.row() > e2.row()) {
		return false;
	} else if (e1.column() < e2.column()) {
		return true;
	} else if (e1.column() > e2.column()) {
		return false;
	} else {
		return false;
	}
}

void copyToClipboard(QTableView* table, QClipboard* clipboard) {
	QVector<QModelIndex> vect = table->selectionModel()->selectedIndexes().toVector();
	qSort(vect.begin(), vect.end(), lessThan);

	QString ser;

	int row = -1;
	foreach (QModelIndex index, vect) {
		if (row == -1) {
			row = index.row();
		} else if (row != index.row()) {
			ser = ser % '\n';
			row = index.row();
		} else if (index.row() != 0 || index.column() != 0) {
			ser = ser % '\t';
		}

		ser = ser % table->model()->data(index).toString();
	}

	clipboard->setText(ser);
}

void pastFromClipboard(QTableView* table, QClipboard *clipboard) {
	QVector<QModelIndex> vect = table->selectionModel()->selectedIndexes().toVector();
	qSort(vect.begin(), vect.end(), lessThan);

	int colCount = 0;

	QString ser = clipboard->text();
	QList<QVariant> values;

	QStringList lines = ser.split('\n');
	foreach (QString line, lines) {

		QStringList items = line.split('\t');
		foreach (QString item, items) {
			values.append(QVariant(item));
		}
		if (colCount == 0) colCount = items.count();
	}

	int selCount = vect.count();
	int valCount = values.count();
	int size = qMax(selCount, valCount);

	int row = 0, column = 0;
	for (int i = 0; i < size; i++) {
		if (selCount == 0) {
			row = i / colCount;
			column = i % colCount;
		}	else if (i < selCount) {
			row = vect.at(i).row();
			column = vect.at(i).column();
		} else {
			row = vect.at(i % selCount).row() + i / colCount;
			column = vect.at(i % selCount).column();
		}

		QModelIndex index = table->model()->index(row, column);
		QVariant value = values.at(i % valCount);
		if (!table->itemDelegate(index)->inherits(ReadonlyDelegate::staticMetaObject.className())) {
			table->model()->setData(index, value);
		}
	}

}

// column
int addColumn(QStandardItemModel *model, const QString &name) {
	int index = model->columnCount();

	model->appendColumn(QList<QStandardItem *>());
	model->setHorizontalHeaderItem(index, new QStandardItem(name));

	return index;
}

void DataGrid::addBoolColumn(const QString &name) {
	int index = addColumn(this->model, name);
	ui->tableView->setItemDelegateForColumn(index, new CheckboxDelegate(ui->tableView));
}

void DataGrid::addIntColumn(const QString &name, int minimum, int maximum) {
	int index = addColumn(this->model, name);
	ui->tableView->setItemDelegateForColumn(index, new SpinboxDelegate(minimum, maximum, ui->tableView));
}

void DataGrid::addStringColumn(const QString &name) {
	addColumn(this->model, name);
}

void DataGrid::addReadonlyColumn(const QString &name) {
	int index = addColumn(this->model, name);
	ui->tableView->setItemDelegateForColumn(index, new ReadonlyDelegate(ui->tableView));
}

void DataGrid::addComboColumn(const QString &name, const QList<QVariant> &texts, const QList<QVariant> &values, bool readonly) {
	int index = addColumn(this->model, name);
	ui->tableView->setItemDelegateForColumn(index, new ComboboxDelegate(texts, values, readonly, ui->tableView));
}

void DataGrid::addComboColumn(const QString &name, const QList<QVariant> &texts, const QList<QVariant> &values, const QList<QVariant> &colors, bool readonly) {
	int index = addColumn(this->model, name);
	ui->tableView->setItemDelegateForColumn(index, new ComboboxDelegate(texts, values, colors, readonly, ui->tableView));
}

void DataGrid::addButtonColumn(const QString &name, const QString &text) {
	int index = addColumn(this->model, name);
	ui->tableView->setItemDelegateForColumn(index, new ButtonDelegate(text, this, ui->tableView));
}

void DataGrid::setColumnHidden(int id, bool hide)
{
	ui->tableView->setColumnHidden(id, hide);
}

void DataGrid::setColumnWidth(int id, int width)
{
	ui->tableView->setColumnWidth(id, width);
}

// values
QVariant DataGrid::getValue(int row, int column) {
	return this->model->data(this->model->index(row, column));
}

void DataGrid::setValue(int row, int column, QVariant &value) {
	isWriting = true;
	this->model->setData(this->model->index(row, column), value);

	applyFilter();
	applySortOrder();
	isWriting = false;
}

int DataGrid::getRow(const QString &key)
{
	if (!keyIndex.contains(key)) return -1;
	return keyIndex.value(key);
}

int DataGrid::getColumn(const QString &name)
{
	for (int i = 0; i < this->model->columnCount(); i++) {
		if (this->model->headerData(i, Qt::Vertical).toString() == name) {
			return i;
		}
	}
	return -1;
}

void DataGrid::setCellForegroundColor(int row, int column, const QString &color)
{
	QStandardItem* item = model->item(row, column);
	if (item != NULL)
		item->setData(QColor(color), Qt::ForegroundRole);
}

void DataGrid::setRowForegroundColor(int row, const QString &color)
{
	for (int column = 0; column < this->model->columnCount(); column++)
		setCellForegroundColor(row, column, color);
}

void DataGrid::setColumnForegroundColor(int column, const QString &color)
{
	for (int row = 0; row < this->model->rowCount(); row++)
		setCellForegroundColor(row, column, color);
}

void DataGrid::setCellBackgroundColor(int row, int column, const QString & color)
{
	QStandardItem* item = model->item(row, column);
	if (item != NULL)
		item->setData(QColor(color), Qt::BackgroundRole);
}

void DataGrid::setRowBackgroundColor(int row, const QString & color)
{
	for (int column = 0; column < this->model->columnCount(); column++)
		setCellBackgroundColor(row, column, color);
}

void DataGrid::setColumnBackgroundColor(int column, const QString & color)
{
	for (int row = 0; row < this->model->rowCount(); row++)
		setCellBackgroundColor(row, column, color);
}

void DataGrid::addContextualMenu(const QString &name)
{
	contextualMenus.append(name);
}

QList<QVariant> DataGrid::getColumnValues(int column) {
	this->invalidate();

	QList<QVariant> values = QList<QVariant>();
	for (int row = 0; row < this->model->rowCount(); row++) {
		values << this->model->data(this->model->index(row, column));
	}
	return values;
}

void DataGrid::setColumnValues(int column, const QList<QVariant> &values) {

	isWriting = true;


	int row = 0;
	foreach (QVariant value, values) {

		if (this->model->rowCount() <= row) {
			this->model->appendRow(QList<QStandardItem *>());
		}
		this->model->setData(this->model->index(row, column), value);
		row++;
	}

	applyFilter();
	applySortOrder();

	isWriting = false;

}

QList< QList<QVariant> > DataGrid::getValues() {
	this->invalidate();

	QList<QList<QVariant> > values;

	for (int column = 0; column < this->model->columnCount(); column++) {

		values << this->getColumnValues(column);
	}

	return values;
}

QList<QVariant> DataGrid::getRowValues(int row) {
	this->invalidate();

	QList<QVariant> values = QList<QVariant>();
	for (int column = 0; column < this->model->columnCount(); column++) {
		values << this->model->data(this->model->index(row, column));
	}
	return values;
}

void DataGrid::addRowValues(const QList<QVariant> &values) {

	isWriting = true;

	int row = 0;

	if (this->keyColumn != -1) {
		const QString keyData = values.at(this->keyColumn).toString();

		if (this->keyIndex.contains(keyData)) {
			row = this->keyIndex.value(keyData);

		} else {
			this->model->appendRow(QList<QStandardItem *>());
			row = this->model->rowCount()-1;
			this->keyIndex.insert(keyData, row);
		}
	} else {
		this->model->appendRow(QList<QStandardItem *>());
		row = this->model->rowCount()-1;
	}

	int column = 0;
	foreach (QVariant value, values) {
		this->model->setData(this->model->index(row, column), value);
		column++;
	}

	applyFilter();
	applySortOrder();

	isWriting = false;

}

void DataGrid::setKeyColumn(int keyColumn) {
	this->keyColumn = keyColumn;
}

void DataGrid::setValues(QList<QList<QVariant> > values) {

	isWriting = true;

	this->model->clear();
	foreach (QList<QVariant> line, values) {
		this->addRowValues(line);
	}

	applyFilter();
	applySortOrder();

	isWriting = false;

}

// enable
void DataGrid::setCellEnabled(int row, int column, bool enable) {
	QStandardItem *item = this->model->item(row, column);

	if (enable) {
		item->setData(QBrush(Qt::black), Qt::ForegroundRole);
		item->setFlags(item->flags() | (Qt::ItemIsEditable));
	} else {
		item->setData(QBrush(Qt::gray), Qt::ForegroundRole);
		item->setFlags(item->flags() & (~Qt::ItemIsEditable));
	}

}

// style
void DataGrid::setStyleSheetFile(const QString &filename)
{
	QFile qss(filename);
	qss.open(QFile::ReadOnly);
	setStyleSheet(QLatin1String(qss.readAll()));

}

void DataGrid::setStyleSheet(const QString &styleSheet)
{
	QWidget::setStyleSheet(styleSheet);
}

// selection
QList<QVariant> DataGrid::getSelectedIndex() {

	QList<QVariant> list;
	foreach (QModelIndex index, ui->tableView->selectionModel()->selectedIndexes()) {
		list.append(index.row());
		list.append(index.column());
	}

	return list;
}

QList<QVariant> DataGrid::getSelectedRows() {
	QList<QVariant> list;
	foreach (QModelIndex index, ui->tableView->selectionModel()->selectedRows()) {
		list.append(index.row());
	}

	return list;
}

QList<QVariant> DataGrid::getSelectedColumns() {
	QList<QVariant> list;
	foreach (QModelIndex index, ui->tableView->selectionModel()->selectedColumns()) {
		list.append(index.column());
	}

	return list;
}

void DataGrid::clearSelection() {
	ui->tableView->selectionModel()->clear();
}

// tools
void DataGrid::copy() {
	copyToClipboard(ui->tableView, QApplication::clipboard());
}

void DataGrid::paste() {
	pastFromClipboard(ui->tableView, QApplication::clipboard());
}

// private
void DataGrid::invalidate() {
	ui->tableView->setDisabled(true);
	ui->tableView->setDisabled(false);
}

// override
void DataGrid::keyPressEvent(QKeyEvent *key) {
	if (key->matches(QKeySequence::Copy)) {
		copyToClipboard(ui->tableView, QApplication::clipboard());
	} else if (key->matches(QKeySequence::Paste)) {
		pastFromClipboard(ui->tableView, QApplication::clipboard());
	}
}

// event
void DataGrid::fireGridButtonClicked(int row, int column) {
	QModelIndex index = getModelIndexFromProxy(row, column);
	emit gridButtonClicked(index.row(), index.column());
}

// slot
void DataGrid::onDataChanged(const QModelIndex &topLeft, const QModelIndex &bottomRight) {
	int top = topLeft.row();
	int bottom = bottomRight.row();
	int left = topLeft.column();
	int right = bottomRight.column();

	if (!isWriting) {
		for (int row = top; row <= bottom; row++) {
			for (int column = left; left <= right; left++) {
				QModelIndex index = getModelIndexFromProxy(row, column);

				QVariant data = this->model->data(index);
				emit cellValueChanged(index.row(), index.column(), data);
			}
		}
	}
}

void DataGrid::onCustomMenuRequested(QPoint pos){

	currentPosition = pos;
	QModelIndex index = ui->tableView->indexAt(pos);
	currentValue = this->proxyModel->data(index);

	if (QStyledItemDelegate* delegate = dynamic_cast<QStyledItemDelegate*>(ui->tableView->itemDelegateForColumn(index.column()))) {
		currentText = delegate->displayText(currentValue, QLocale::English);
	} else {
		currentText = currentValue.toString();
	}

	currentColumnIndex=ui->tableView->columnAt(pos.x());
	currentColumnName=proxyModel->headerData(currentColumnIndex, Qt::Horizontal).toString();

	QMenu* menu = new QMenu(this);

	if (contextualMenus.count() > 0) {
		for (int i = 0; i < contextualMenus.count(); i++) {
			createMenu(menu, contextualMenus.at(i), FilterAction::CUSTOM);
		}
		menu->addSeparator();
	}

	createMenu(menu, QString("Clear filter"), FilterAction::CLEAR);

	if (index.isValid()) {
		menu->addSeparator();
		createMenu(menu, QString("Add equals - %1").arg(currentText), FilterAction::EQUALS);
		createMenu(menu, QString("Add not equals - %1").arg(currentText), FilterAction::NOT_EQUALS);
	}

	connect(menu, SIGNAL(triggered(QAction*)), SLOT(onMenuAction(QAction*)));

	menu->popup(ui->tableView->viewport()->mapToGlobal(pos));

}

void DataGrid::onSortOrderChanged(int, Qt::SortOrder) {
	if (!isWriting) {
		emit sortChanged();
	}
}

void DataGrid::createMenu(QMenu *menu, const QString& name, FilterAction::Enum filterAction) {
	QAction* action = new QAction(name, menu);
	action->setData(filterAction);
	menu->addAction(action);
}

QModelIndex DataGrid::getModelIndexFromProxy(int row, int col) {
	QModelIndex proxyIndex = this->proxyModel->index(row, col);
	return this->proxyModel->mapToSource(proxyIndex);
}

void DataGrid::onMenuAction(QAction* action) {

	FilterAction::Enum filterAction = (FilterAction::Enum)action->data().toInt();

	if (filterAction == FilterAction::CUSTOM) {

		QModelIndex index = getModelIndexFromProxy(ui->tableView->rowAt(currentPosition.y()), ui->tableView->columnAt(currentPosition.x()));
		emit contextualMenuClicked(action->text(), index.row(), index.column());

	} else {

		if (filterAction == FilterAction::CLEAR) {

			filter->clear();
			clearFilterWidget();
			emit filterChanged();

		} else {
			FilterElement* query = NULL;

			if (filterAction == FilterAction::EQUALS) {
				query = new Equals(currentColumnIndex, currentColumnName, currentValue, currentText, action->menu());
			} else if (filterAction == FilterAction::NOT_EQUALS) {
				query = new NotEquals(currentColumnIndex, currentColumnName, currentValue, currentText, action->menu());
			}

			addFilter(query);
			emit filterChanged();

		}

		applyFilter();
	}
}

void DataGrid::onFilterClosed(FilterWidget* sender)
{
	filter->remove(sender->filter());
	applyFilter();
	emit filterChanged();

	ui->filterStack->layout()->removeWidget(sender);
	applySortOrder();
	delete sender;
}

void DataGrid::addFilter(FilterElement *query)
{

	if (filter->append(query)) {
		query->setParent(this);
		addFilterWidget(query->name(), query);
	} else {
		delete query;
	}

}

void DataGrid::clearFilterWidget() {
	QLayoutItem* item;
	while ( ( item = ui->filterStack->layout()->takeAt( 0 ) ) != NULL ) {
		delete item->widget();
		delete item;
	}
	ui->filterStack->layout()->invalidate();

	applySortOrder();

}

void DataGrid::addFilterWidget(const QString&, const FilterElement *filter) {
	FilterWidget* filterWidget = new FilterWidget(filter, ui->filterStack);
	ui->filterStack->layout()->addWidget(filterWidget);
	connect(filterWidget, SIGNAL(closed(FilterWidget*)), SLOT(onFilterClosed(FilterWidget*)));
}

void DataGrid::applyFilter() {
	proxyModel->applyFilter(this->filter);
}

void DataGrid::applySortOrder() {
	int column = ui->tableView->horizontalHeader()->sortIndicatorSection();
	Qt::SortOrder order = ui->tableView->horizontalHeader()->sortIndicatorOrder();

	ui->tableView->sortByColumn(column, order);
}

QString DataGrid::buildXmlSettings()
{
	QDomDocument dom = QDomDocument();

	QDomElement root = dom.createElement("datagridSettings");
	dom.appendChild(root);

	//

	QDomElement domSort = dom.createElement("sort");
	domSort.setAttribute("column", ui->tableView->horizontalHeader()->sortIndicatorSection());
	domSort.setAttribute("order", ui->tableView->horizontalHeader()->sortIndicatorOrder());
	root.appendChild(domSort);

	QDomElement domFilter = dom.createElement("filter");
	root.appendChild(domFilter);

	filter->buildXmlSettings(dom, domFilter);

	return dom.toString();
}

void DataGrid::applyXmlSettings(const QString& xml)
{
	QDomDocument dom = QDomDocument();
	dom.setContent(xml);

	QDomElement root = dom.documentElement();

	QDomElement domSort = root.elementsByTagName("sort").at(0).toElement();
	QDomAttr sortColumn = domSort.attributeNode("column");
	QDomAttr sortOrder = domSort.attributeNode("order");

	this->filter->clear();
	this->clearFilterWidget();
	QDomElement domFilter = root.elementsByTagName("filter").at(0).toElement();
	QDomElement andListFilter = domFilter.elementsByTagName("AndList").at(0).toElement();

	for (int i = 0; i < andListFilter.childNodes().count(); i++) {
		QDomElement element = andListFilter.childNodes().at(i).toElement();
		QString tagName = element.tagName();
		int columnIndex = element.attributeNode("columnIndex").value().toInt();
		QString columnName = element.attributeNode("columnName").value();
		QVariant compare = element.attributeNode("compare").value();
		QString compareText = element.attributeNode("compareText").value();

		if (tagName == QString("Equals")) {
			addFilter(new Equals(columnIndex, columnName, compare, compareText, this));
		} else if (tagName == QString("NotEquals")) {
			addFilter(new NotEquals(columnIndex, columnName, compare, compareText, this));
		}
	}

	applyFilter();
	ui->tableView->horizontalHeader()->setSortIndicator(sortColumn.value().toInt(), (Qt::SortOrder)sortOrder.value().toInt());
	applySortOrder();
}
