#include "FlowLayout.h"

FlowLayout::FlowLayout(QWidget *parent, int margin, int hSpacing, int vSpacing)
	: QLayout(parent), m_hSpace(hSpacing), m_vSpace(vSpacing)
{
	setContentsMargins(margin, margin, margin, margin);

	hOrientation = FlowLayout::LeftToRight;
	vOrientation = FlowLayout::TopDown;
}

FlowLayout::FlowLayout(int margin, int hSpacing, int vSpacing)
	: QLayout(), m_hSpace(hSpacing), m_vSpace(vSpacing)
{
	setContentsMargins(margin, margin, margin, margin);

	hOrientation = FlowLayout::LeftToRight;
	vOrientation = FlowLayout::TopDown;
}

FlowLayout::~FlowLayout()
{
	QLayoutItem *item;
	while ((item = takeAt(0)))
		delete item;
}

void FlowLayout::addItem(QLayoutItem *item)
{
	itemList.append(item);
}

int FlowLayout::horizontalSpacing() const
{
	if (m_hSpace >= 0) {
		return m_hSpace;
	} else {
		return smartSpacing(QStyle::PM_LayoutHorizontalSpacing);
	}
}

int FlowLayout::verticalSpacing() const
{
	if (m_vSpace >= 0) {
		return m_vSpace;
	} else {
		return smartSpacing(QStyle::PM_LayoutVerticalSpacing);
	}
}

int FlowLayout::count() const
{
	return itemList.size();
}

QLayoutItem *FlowLayout::itemAt(int index) const
{
	return itemList.value(index);
}

QLayoutItem *FlowLayout::takeAt(int index)
{
	if (index >= 0 && index < itemList.size())
		return itemList.takeAt(index);
	else
		return 0;
}

Qt::Orientations FlowLayout::expandingDirections() const
{
	return 0;
}

bool FlowLayout::hasHeightForWidth() const
{
	return true;
}

int FlowLayout::heightForWidth(int width) const
{
	int height = doLayout(QRect(0, 0, width, 0), true);
	return height;
}

void FlowLayout::setGeometry(const QRect &rect)
{
	QLayout::setGeometry(rect);
	doLayout(rect, false);
}

QSize FlowLayout::sizeHint() const
{
	return minimumSize();
}

QSize FlowLayout::minimumSize() const
{
	QSize size;
	QLayoutItem *item;
	foreach (item, itemList)
		size = size.expandedTo(item->minimumSize());

	size += QSize(2*margin(), 2*margin());
	return size;
}

int FlowLayout::getInitialX(const QRect& rect) const
{
	return hOrientation == FlowLayout::LeftToRight ? rect.x() : rect.right();
}

int FlowLayout::getInitialY(const QRect& rect) const
{
	return vOrientation == FlowLayout::TopDown ? rect.y() : rect.bottom();
}

int FlowLayout::getNextX(const QSize& sizeHint, int x, int spaceX) const
{
	return hOrientation == FlowLayout::LeftToRight ?
				x + sizeHint.width() + spaceX :
				x - (sizeHint.width() + spaceX);
}

int FlowLayout::getNextY(int lineHeight, int spaceY) const
{
	return vOrientation == FlowLayout::TopDown ?
				lineHeight + spaceY :
				- (lineHeight + spaceY);
}

int FlowLayout::getGeometryX (const QSize& sizeHint, int x) const
{
	return hOrientation == FlowLayout::LeftToRight ? x : x - sizeHint.width();
}

int FlowLayout::getGeometryY (const QSize& sizeHint, int y) const
{
	return vOrientation == FlowLayout::TopDown ? y : y - sizeHint.height();
}

bool FlowLayout::evalNewLine(const QRect& rect, int nextX, int spaceX) const
{
	return hOrientation == FlowLayout::LeftToRight ?
				nextX - spaceX > rect.right() :
				nextX + spaceX < rect.x();
}

int FlowLayout::doLayout(const QRect &rect, bool testOnly) const
{
	int left, top, right, bottom;
	getContentsMargins(&left, &top, &right, &bottom);
	QRect effectiveRect = rect.adjusted(+left, +top, -right, -bottom);
	int x = getInitialX(effectiveRect);
	int y = getInitialY(effectiveRect);
	int lineHeight = 0;
	QLayoutItem *item;
	foreach (item, itemList) {
		QWidget *wid = item->widget();
		int spaceX = horizontalSpacing();
		if (spaceX == -1)
			spaceX = wid->style()->layoutSpacing(
						QSizePolicy::PushButton, QSizePolicy::PushButton, Qt::Horizontal);
		int spaceY = verticalSpacing();
		if (spaceY == -1)
			spaceY = wid->style()->layoutSpacing(
						QSizePolicy::PushButton, QSizePolicy::PushButton, Qt::Vertical);
		int nextX = getNextX(item->sizeHint(), x, spaceX);
		if (evalNewLine(effectiveRect, nextX, spaceX) && lineHeight > 0) {
			x = getInitialX(effectiveRect);
			y += getNextY(lineHeight, spaceY);
			nextX = getNextX(item->sizeHint(), x, spaceX);
			lineHeight = 0;
		}

		if (!testOnly) {
			int gX = getGeometryX(item->sizeHint(), x);
			int gY = getGeometryY(item->sizeHint(), y);
			item->setGeometry(QRect(QPoint(gX, gY), item->sizeHint()));
		}

		x = nextX;
		lineHeight = qMax(lineHeight, item->sizeHint().height());
	}
	return y + lineHeight - rect.y() + bottom;
}

int FlowLayout::smartSpacing(QStyle::PixelMetric pm) const
{
	QObject *parent = this->parent();
	if (!parent) {
		return -1;
	} else if (parent->isWidgetType()) {
		QWidget *pw = static_cast<QWidget *>(parent);
		return pw->style()->pixelMetric(pm, 0, pw);
	} else {
		return static_cast<QLayout *>(parent)->spacing();
	}
}
