#include "ComboboxDelegate.hxx"
#include <QApplication>
#include <QPainter>
#include <QComboBox>
#include <QDebug>

ComboboxDelegate::ComboboxDelegate(QList<QVariant> texts, QList<QVariant> values, bool readonly, QObject *parent)
	: QStyledItemDelegate(parent), readonly(readonly), texts(texts), values(values)
{
}

ComboboxDelegate::ComboboxDelegate(QList<QVariant> texts, QList<QVariant> values, QList<QVariant> colors, bool readonly, QObject *parent)
	: QStyledItemDelegate(parent), readonly(readonly), texts(texts), values(values), colors(colors)
{
}

QColor blendColors(const QColor& a, const QColor& b, float ratio) {

	QColor c(Qt::black);
	c.setRed(a.red() * (1-ratio) + b.red() * ratio);
	c.setGreen(a.green() * (1-ratio) + b.green() * ratio);
	c.setBlue(a.blue() * (1-ratio) + b.blue() * ratio);
	return c;
}

QColor blendColorsOrBackground(const QColor& a, const QColor& b, float ratio, const QColor& background) {
	if (b == background)
		return QColor(a);
	else
		return blendColors(a, b, ratio);
}

void ComboboxDelegate::paint(QPainter *painter,
														 const QStyleOptionViewItem &option,
														 const QModelIndex &index) const {

	//painter->save();

	QVariant value = index.model()->data(index);
	int i = this->values.indexOf(value);

	if (i == -1) {
		i = 0;
	}

	QColor background = index.model()->data(index, Qt::BackgroundRole).value<QColor>();

	if (colors.count() > i) {
		QColor color = QColor(this->colors.at(i).toString());
		if (background.isValid())
			color = blendColorsOrBackground(color, background, 0.8f, option.palette.background().color());

		painter->fillRect(option.rect, color);
		if (color.lightness() < 100) {
			painter->setPen(option.palette.highlightedText().color());
		} else {
			painter->setPen(option.palette.text().color());
		}
		painter->setBrush(option.palette.highlightedText());
	} else {
		if (background.isValid()) {
			painter->fillRect(option.rect, background);
		}
	}

	QColor foreground = index.model()->data(index, Qt::ForegroundRole).value<QColor>();
	if (foreground.isValid()) painter->setPen(foreground);

	QStyleOptionComboBox combo;
	combo.rect = option.rect;
	combo.currentText = this->texts.at(i).toString();
	combo.state = option.state | QStyle::State_Enabled;

	QApplication::style()->drawControl(QStyle::CE_ComboBoxLabel, &combo, painter);
}

QWidget *ComboboxDelegate::createEditor(QWidget *parent,
																				const QStyleOptionViewItem &/* option */,
																				const QModelIndex &/* index */) const {

	if (readonly == true) return 0;

	QComboBox *editor = new QComboBox(parent);

	for (int i = 0; i<this->texts.count(); i++) {
		editor->addItem(this->texts.at(i).toString(), this->values.at(i));
	}

	connect(editor, SIGNAL(currentIndexChanged(int)), this, SLOT(onCurrentStateChanged(int)));
	editor->installEventFilter(const_cast<ComboboxDelegate*>(this));

	return editor;
}

void ComboboxDelegate::setEditorData(QWidget *editor,
																		 const QModelIndex &index) const {

	QVariant value = index.model()->data(index);

	QComboBox *comboBox = static_cast<QComboBox*>(editor);
	int i = comboBox->findData(value);
	if (i == -1) {
		i = 0;
	}
	comboBox->setCurrentIndex(i);
}

void ComboboxDelegate::setModelData(QWidget *editor,
																		QAbstractItemModel *model,
																		const QModelIndex &index) const {

	QComboBox *comboBox = static_cast<QComboBox*>(editor);
	QVariant value = comboBox->itemData(comboBox->currentIndex());
	model->setData(index, value);
}

void ComboboxDelegate::updateEditorGeometry(QWidget *editor,
																						const QStyleOptionViewItem &option,
																						const QModelIndex &/* index */) const {
	editor->setGeometry(option.rect);
}

void ComboboxDelegate::onCurrentStateChanged(int)
{
	QWidget *editor = qobject_cast<QWidget*>(sender());
	emit commitData(editor);
}

QString ComboboxDelegate::displayText(const QVariant &value, const QLocale &) const
{
	int i = this->values.indexOf(value);

	if (i == -1) {
		i = 0;
	}

	return this->texts.at(i).toString();
}
