//*********************************************************
// Copyright  2018 by FM . Please copy and distribute. "Share the fun"
//*********************************************************
// Description:
//  These functions are used to scan through your
//  software looking for the magix word 'module' and
//  'prefix' (just below this header)
//
//  Any file containing these keywords is added to our 
//  'model' of your application.
//  The keyword is just used to assign the source file
//  to any group.
//
//  Example:
//  You may have N source files that together form
//  the module 'alarmbanner'
//
// ********************************************************

// Kleur van een node gaat via : http://wingraphviz.sourceforge.net/wingraphviz/language/node_color.htm


global mapping g_ta_Software_mModules;

//module:TopAce_SoftwareScan
//prefix:ta_Software_

// ********************************************************
// Description:
//   Will clear the global data before we start
//   scanning for files
//
// Usage:
//   ta_Software_Init();
//
// Returns:
//   nothing 
// ********************************************************

void ta_Software_Init()
{
  
  g_ta_GlobalData.Reset();

}

// ********************************************************
// Description:
//  A number of files have been found.
//  The names and properties have been stored
//  in our global data.
//  This function will get that set of files
//
// Usage:
//   dyn_string dstrFiles = ta_Software_GetFiles();
//
// Returns:
//   A sorted list of files that were found during scanningnothing 
// ********************************************************

dyn_string ta_Software_GetFiles()
{
  dyn_string dstrModules;
  dyn_string dstrFiles;
  
  g_ta_GlobalData.GetFilesAndModules( dstrModules, dstrFiles );
  dynSort( dstrFiles );
  
  return dstrFiles;
}

// ********************************************************
// Description:
//  This is the function that scans the folders
//  looking for files that contain the magic module
//  and/or function prefix
//
//  When we have all module and prefixes then we make a
//  second pass to determine the relationships
//
// Usage:
//   ta_Software_Scan();
//
// Returns:
//   nothing 
// ********************************************************

void ta_Software_Scan()
{
  dyn_string dstrPrefix;
  dyn_string dstrModules;
  
  // The first run will scan for module name and function prefix
  ta_Software_ScanRun1Folder( getPath( SCRIPTS_REL_PATH ), "scripts/" );
  ta_Software_ScanRun1Folder( getPath( PANELS_REL_PATH  ), "panels/"  );
  
  // What prefix and modules could we find
  g_ta_GlobalData.GetPrefixAndModule( dstrPrefix, dstrModules ); 
  
  // We now know all modules and function prefixes and can make
  // the second run
  ta_Software_ScanRun2Folder( getPath( SCRIPTS_REL_PATH ), "scripts/", dstrPrefix, dstrModules );
  ta_Software_ScanRun2Folder( getPath( PANELS_REL_PATH  ), "panels/" , dstrPrefix, dstrModules );

}

// ********************************************************
// Description:
//  Will do the first pass. Looking
//  for the module and function prefix keyword.
//  The functin is recurive and will
//  go into a subfolder
//
// Usage:
//   ta_Software_ScanFolder( PANELS_REL_PATH, "panels/" );
//
// Returns:
//   nothing 
// ********************************************************


void ta_Software_ScanRun1Folder(
  string strFolder,    // Scan an absolute(!) folder ( function is used recursively)
  string strRelative   // Relative path
)
{
  //Look for files
  dyn_string dstrFiles = getFileNames( strFolder, "*", FILTER_FILES );
  
  for( int t =1; t <= dynlen( dstrFiles ); t++)
  {
    string strFile = dstrFiles[t];
    
    // Ignore the *.bak files
    if( strpos( strFile, ".bak" ) <= 0 )
    {
      // This is nor a bak file so scan the file
      ta_Software_ScanRun1File( strFolder, strRelative, strFile );
    }
  }
  
  
  // now look for subfolders
  // so that we can scan this subfolder recursively
  dyn_string dstrFolders = getFileNames( strFolder, "*", FILTER_DIRS );

  for( int t = 1; t <= dynlen( dstrFolders ); t++)
  {
    string strSubFolder = dstrFolders[t];
    
    // Do not scan the folder ".", ".." and the temporary folder with XML converted panels "TopAceTempXML"
    if( (strSubFolder != ".") && (strSubFolder != ".." ) && (strSubFolder != "TopAceTempXML") )
    {
      // Determine the relative path
      string strSubRelative;

      if( strlen( strRelative ))
      {
        strSubRelative = strRelative + strSubFolder +"/";
      }      
      else
      {
        strSubRelative = strSubFolder +"/";
      }
      
      //Now go in recursively
      ta_Software_ScanRun1Folder( strFolder + strSubFolder + "/", strSubRelative );
    }
  }
  
}

// ********************************************************
// Description:
//   This function will scan a specific file
//   It will look for the magic word
//   "//" + "module:' 
//
//   When we find the keyword then we scan the file in more detail
//
// Usage:
//   ta_Software_ScanFile( strFile, strRelatiefPath );
//
// Returns:
//   nothing 
// ********************************************************


void ta_Software_ScanRun1File(
  string strAbsoluteFolder,   // Absolute folder  
  string strRelative,         // Relative path pad
  string strFile              // Het bestand
)
{
  string strContent;
  string strModule;
  
  // Determine the absolute path to the file
  string strFileAbsolute = strAbsoluteFolder + strFile;
  
  // Read the entire file (we're about to look for the module word)
  fileToString( strFileAbsolute, strContent );
  
  // Can we find the magic module word ?
  if( strpos( strContent, "//" + "module:") >= 0 )
  {
    // Split the file into lines
    dyn_string dstrLines = strsplit( strContent, "\n" );
    
    for( int t= 1; t <= dynlen( dstrLines ); t++)
    {
      string strLine = dstrLines[t];
      
      if( strpos( strLine, "//" + "module:" ) >= 0 )
      {
    
        // They can use a mnu item to skip files that contain the magic word 'TopAce'
        if( !TopAceMenu_Option_DoGraphTopAce() )
        {
          if( strpos( strLine, "TopAce" ) >= 0 )
          {
            // Yes, the file contains "TopAce' so 
            // we just exit here !
            return;
          }
        }
            
        // This line contains the magix word module
        // so we isolate the module name
        int iPos = strpos( strLine, ":" );
        strModule = substr( strLine, iPos + 1 );
   
        ta_Software_AddModule( strModule );
        ta_Software_AddFileToModule( strModule, strRelative + strFile );
      }
      
      if( strpos( strLine, "//" + "prefix:" ) == 0 )
      {
        // This line contains the magic word
        // so isolate the function prefix
        int iPos = strpos( strLine, ":" );
        string strPrefix = substr( strLine, iPos + 1 );
   
        ta_Software_AddModule( strModule );
        ta_Software_AddPrefixTeModule( strModule, strPrefix );
      }      
    }
  }
  
}



// ********************************************************
// Description:
//   This function scans a folder and then calls
//   ta_Software_ScanRun2File for each file
//   
//   Will also look for subfolders and then call
//   this function recursively
//
// Usage:
//   ta_Software_ScanRun2Folder( PANELS_REL_PATH, "panels/" );
//
// Returns:
//   nothing 
// ********************************************************


void ta_Software_ScanRun2Folder(
  string strFolder,        // Folder to scan (recursively)
  string strRelative,      // Relative path
  dyn_string dstrPrefix,   // a function prefix
  dyn_string dstrModules   // and the module where it was found
)
{
  // Look for files ( so that we can then isolate the keywords)
  dyn_string dstrFiles = getFileNames( strFolder, "*", FILTER_FILES );
  
  for( int t =1; t <= dynlen( dstrFiles ); t++)
  {
    string strFile = dstrFiles[t];
    
    // We do't want to scan bak files
    if( strpos( strFile, ".bak" ) <= 0 )
    {
      // No, thsi is not a BAK file
      ta_Software_ScanRun2File( strFolder, strRelative, strFile,  dstrPrefix, dstrModules );
    }
  }
  
  
  // Now look for folders
  // and then go in recursively
  dyn_string dstrFolders = getFileNames( strFolder, "*", FILTER_DIRS );

  for( int t = 1; t <= dynlen( dstrFolders ); t++)
  {
    string strSubFolder = dstrFolders[t];
    
    // Do not scan the folder ".", ".." and the temporary folder with XML converted panels "TopAceTempXML"
    if( (strSubFolder != ".") && (strSubFolder != ".." ) && (strSubFolder != "TopAceTempXML") )
    {
      // Determine relative path
      string strSubRelative;

      if( strlen( strRelative ))
      {
        strSubRelative = strRelative + strSubFolder +"/";
      }      
      else
      {
        strSubRelative = strSubFolder +"/";
      }
      
      // Now call this function recursively
      ta_Software_ScanRun2Folder( strFolder + strSubFolder + "/", strSubRelative, dstrPrefix, dstrModules );
    }
  }
  
}

// ********************************************************
// Description:
//   We hebben in een eerdere slag een reeks met prefixes bepaald
//   nu kunnen we kijken of die prefix ergens wordt gebruikt
//
// Usage:
//   ta_Software_ScanRun2File( strFile, strRelatiefPath );
//
// Returns:
//   nothing 
// ********************************************************

void ta_Software_ScanRun2File(
  string strAbsoluteFolder,   // Absolute folder  
  string strRelative,         // Relative path
  string strFile,          // Het bestand
  dyn_string dstrPrefix,      // a function prefix
  dyn_string dstrModules      // and the module where it was found  
)
{
  string strContent;

  

  // What module owns this file ?
  string strModuleFile = ta_Software_FindModuleFile( strRelative + strFile );
  
  // Determine the absolute path of the file
  string strFileAbsolute = strAbsoluteFolder + strFile;
  
  // Now read the entire file and split into lines
  fileToString( strFileAbsolute, strContent );
  dyn_string dstrLines = strsplit( strContent, "\n" );

  // Now look for ther function prefixes
  for( int t = 1; t <= dynlen( dstrLines ); t++)
  {
    string strLine = dstrLines[t];
    
    for( int i = 1; i <= dynlen( dstrPrefix ); i++)
    {
      // =============================================================
      // Komt de prefix voor in het bestand wat we net bekijken
      if( strpos( strLine, dstrPrefix[i] ) >= 0 )
      {
        
        // we hebben nu een bestand (zelf wellicht deel van een module)
        // die een functie uit een bepaalde module lijkt aan te roepen
        // Dit moeten we in onze 'boekhouding' bijhouden
        
        // en bij welke module hoort de prefix
        string strModulePrefix = dstrModules[i];
        
        if( strModuleFile != strModulePrefix )
        {
          // deze verdient aandacht
          
          if( strModuleFile != "" )
          {
            // Time to keep this relationship
            ta_Software_AddModuleUsesModule( 
              strModuleFile, 
              strModulePrefix, 
              true );            // TRUE -> dit is een stevige relatie (aanroep van een script)
          }
          
          else
          {
            DebugN( "--> Bestand " + strRelative + strFile  + " hoort nog niet bij een module" );
          }
        }  
      }
    }
    
    dyn_string dstrFiles;
    dyn_string dstrModules;
    g_ta_GlobalData.GetFilesAndModules( dstrModules, dstrFiles );
    
    
    for( int i = 1; i <= dynlen( dstrFiles ); i++)
    {
      string strFile = dstrFiles[i];
      string strModuleReference = dstrModules[i];
      
      // when the file is something like : "panels/SampleApplication/typicals/sample.pnl"
      //                                    ^^^^^^^
      // then we need to strip of first part because the word 'panels'
      // wont appear anywhere
      if( strpos( strFile, "panels/" ) == 0 ) 
      {
        int iPos = strpos( strFile, "/" );
        strFile = substr( strFile, iPos + 1 );        
      }
      
      // ======================================================================
      // Is a file from a different module used ?
      // (e.g. a panel loading  a panel
      
      if( strpos( strLine, strFile ) >= 0 )
      {
        DebugN( "--> Bestand " + strFileAbsolute + " gebruikt : " + strFile  );
        // we hebben nu een bestand (een panel of een script)
        // die een referentie naar een ander bestand bevat
        
       
        if( strModuleFile != strModuleReference )
        {
          // deze verdient aandacht
          
          if( strModuleFile != "" )
          {
            // Time to keep this relationship
            ta_Software_AddModuleUsesModule( 
              strModuleFile, 
              strModuleReference, 
              false );              // FALSE this is a panel-to-panel relationship
          }
          
          else
          {
            DebugN( "--> File " + strRelative + strFile  + " is not part of any module" );
          }
        }  
        
      }
      
    }
  }
    
}

// ********************************************************
// Description:
//  Is used during the second scan.
//  We just encountered a file (panel or script)
//  and now need to know to what 'module' it belongs
//
// Usage:
//   string strModule = ta_Software_FindModuleFile( "panels/about.pnl" );
//
// Returns:
//   The module a file belongs to or ""
// ********************************************************

string ta_Software_FindModuleFile(
  string strRelatiefBestand    
)
{
  dyn_string dstrModules = g_ta_GlobalData.GetModuleNames();
  
  for( int t = 1; t <= dynlen( dstrModules ); t++)
  {
    shared_ptr<ta_Module> pModule = g_ta_GlobalData.GetModule( dstrModules[t] );
    
    if( pModule.HasFile( strRelatiefBestand ) )
    {
      // Yes, this module contains that file !
      return dstrModules[t];
    }
  }
  
  return "";
}

// ********************************************************
// Description:
//  We have just discovered a new module
//  and want to add it to the global data
//
// Usage:
//   ta_Software_AddModule( "alarms" );
//
// Returns:
//  Nothing
// ********************************************************

void ta_Software_AddModule(
  string strModule    
)
{
  // add a module to our global data
  g_ta_GlobalData.AddModule( strModule );
}

// ********************************************************
// Description:
//  We found that a module is using another module.
//  This relationship can be :
//
//  1) weak   -> a panel or script opening a panel
//  2) strong -> a panel or script calling a script
//
//  Weak means that a module will not crash when the 
//  used item will crash. E.g. when a panel A contains
//  a panel B then panel A will never crash.
//
//  Strong: When script A calls function B then a bug
//  in function B can make script A crash.
//
// Usage:
//   // Example:
//   // The module 'alarms' is using a panel in module 'alarmbanner'
//   ta_Software_AddModuleUsesModule( "alarms", "alarmbanner", false );
//
// Returns:
//  nothing
// ********************************************************

void ta_Software_AddModuleUsesModule(
  string strModule,                // The module that uses 'strUsesModule'
  string strUsesModule,            // The module being used
  bool bRelationshipScript         // true -> This is a script or panel calling a script
)
{
  // Get a pointer to the specified module
  shared_ptr<ta_Module> pModule = g_ta_GlobalData.GetModule( strModule );
  if( pModule != nullptr )
  {
    // Note that the module is using the specified module
    pModule.AddUses( strUsesModule, bRelationshipScript );
  }  

  // Now add the 'used by'
  shared_ptr<ta_Module> pUsedModule = g_ta_GlobalData.GetModule( strUsesModule );
  if( pUsedModule != nullptr )
  {
    pUsedModule.AddUsedBy( strModule, bRelationshipScript );
  }   
}

// ********************************************************
// Description:
//  A file has been found that is part of a module.
//  We now add this information to the specified module
//  in our global data
//
// Usage:
//   ta_Software_AddFileToModule( "alarms", "alarmbanner.pnl" );
//
// Returns:
//  nothing
// ********************************************************

void ta_Software_AddFileToModule( 
  string strModule,               // de module waartoe dit bestand behoort
  string strRelativeFile       // project relative path
)
{
  // we need absolute path to get file name and date
  string strAbsoluteFile = PROJ_PATH + strRelativeFile;
  
  // get a pointer to this module 
  shared_ptr<ta_Module> pModule = g_ta_GlobalData.GetModule( strModule );
  
  // And add a file to the module
  pModule.AddFile( strRelativeFile );
   
  g_ta_GlobalData.AddFileAndModule( strRelativeFile, strModule );
  
  
}  

// ********************************************************
// Description:
//  we found the prefix keyword in a module
//  and want to retain that information in our
//  global data 'g_ta_GlobalData'
//
// Usage:
//   ta_Software_AddPrefixTeModule( "alarms", "_alarm_" );
//
// Returns:
//  nothing
// ********************************************************

void ta_Software_AddPrefixTeModule( 
  string strModule,               // The module that we found
  string strPrefix                // The prefix keyword that we found
)
{

  // Remember this prefix and module
  g_ta_GlobalData.AddPrefixAndModule( strPrefix, strModule );
} 





// ********************************************************
// Description:
//   Please look at:
//   https://en.wikipedia.org/wiki/DOT_(graph_description_language)
//
//  This function will produce the text file that we can then 
//  turn into a graph using the dot command line tools
//
// Usage:
//   // Will produce a graph including the metrics 
//   // (Metrics are used to clor the items in the graph)
//   ta_Software_ProduceerGraph( false );
//
// Returns:
//   nothing 
// ********************************************************

void ta_Software_ProduceGraphTotal(
  bool bDoMetrics     // TRUE -> metrics have been performed
)
{
  dyn_string dstrTop3;           // we determine the top3 in 'worst' modules
  dyn_string dstrTop3Colors;     // And the colors for these top3 modules
  dyn_dyn_string ddstrDummy;
  
  // The 'dot' file is a set of connections.
  // We do not want to add a connection more than once
  // that is why we remember the connection here
  dyn_string dstrConnections;
  
  // First get a list with modules
  dyn_string dstrModules = g_ta_GlobalData.GetModuleNames();
  
  // When they do not want TopAce in the graph
  // then we remove all modules that contain the magic word 'TopAce'
  if( !TopAceMenu_Option_DoGraphTopAce() )
  {
    for( int t = dynlen( dstrModules ); t > 0; t--)
    {
      string strModule = dstrModules[t];
      
      if( strpos( strModule, "TopAce" ) >= 0 )
      {
        dynRemove( dstrModules, t );
      }
    }
  }
  
  // Determine the top3 colors
  // for the 'worst' modules
  // That is : we want the 'worst' module
  // to have a red color
  if( bDoMetrics )
  {
    ta_Software_DetermineScoreTop3(
      "",  
      dstrModules,        // The first set of modulesl
      makeDynString(),    // second set of modules
      dstrTop3,           // Receives the top 3 of bad modules
      dstrTop3Colors );   // receives the colors for the top 3 bad modules
  }  
  
  // Now show all modules in a graph
  // e.g. add the modules to a text file with their relations
  string strGraph = "digraph software_overview{\n";
  
  for( int t = 1; t <= dynlen( dstrModules ); t++)
  {
    string strModule = dstrModules[t];
    
    if( bDoMetrics )
    {
      string strColor = ta_Software_GetScoreColor( strModule, dstrTop3, dstrTop3Colors );
      if( strlen( strColor ))
      {
        // we use the score to change the color
        // of a module in the graph
        // The 'worst' module becomes red
        strGraph += strModule + "[" + strColor + "];\n";
      }    
    }  
    
    // Get a reference to the module
    shared_ptr<ta_Module> pModule = g_ta_GlobalData.GetModule( strModule );
    
    // What modules are being used
    dyn_string dstrUsed = pModule.GetUses();
    
    for( int i = 1; i <= dynlen( dstrUsed ); i++)
    {
      string strModuleUsed = dstrUsed[i];
      
      bool bRelationScript; 
      int iNumber        ; 
      
      pModule.GetUsesModuleProps( 
        strModuleUsed, 
        bRelationScript,     // TRUE -> script calling script. False -> script/panel containing panel
        iNumber );           // How often is it used
      
      string strConnection;
      
      // When a module is used more than 1 times
      // then we add a label to the line
      // Example: show many times an icon has been used
      string strLabel;
      
      if( iNumber > 1 )
      {
        strLabel = "[label=" + iNumber + "]";
      }
     
      if( bRelationScript )
      {
        // When it is a script->script relation
        // then we draw a normal line
        strConnection = "  " + strModule + " -> " + strModuleUsed + strLabel + ";\n";
      }
      else
      {
        // When it is a script/panel -> panel relation
        // then we draw a dotted line
        strConnection = "  " + strModule + " -> " + strModuleUsed + strLabel + "[style=dotted];\n";
      }  
      
      strGraph += strConnection;
    }
  }
  
  strGraph += "}\n";
  
  
  GRAPH.text =   strGraph;
  
  string strFile = PROJ_PATH + "Data/TopAceOutput/Graph/_index.txt";
  
  // we gaan de graph ook in een bestand opslaan
  file f = fopen( strFile, "w" );
  fputs( strGraph, f );
  fclose( f );
  
  // Now produce the SVG
  ta_Software_ProduceSVG( strFile );
  
  // Now produce the HTML file
  ta_HTML_ProduceHTML( "_Index", dstrModules, true, false, true, false, "", "", ddstrDummy );  
  
}

// ********************************************************
// Description:
//   Call the DOT.exe command line tool
//   to produce the actual graph (in SVG format)
//
// Usage:
//   ta_Software_ProduceSVG();
//
// Returns:
//   nothing 
// ********************************************************

void ta_Software_ProduceSVG(
  string strGraphFile    // What graph file to process    
)
{
  string stdOut;
  string stdErr;
  
  string strDot = "\"C:\\Program Files (x86)\\Graphviz2.38\\bin\\dot.exe\" ";
  
  string strCommand = strDot + "-Kdot -Tsvg -O " + strGraphFile;
  system( strCommand, stdOut, stdErr );
  
  DebugN( strCommand );
  
  // Show result and any errors
  if( strlen( stdOut )) DebugN( stdOut );
  if( strlen( stdErr )) DebugN( stdErr );
}

// ********************************************************
// Description:
//   This function will display the graph
//   called '_index.html'
//
// Usage:
//   ta_Software_ShowSVG();
//
// Returns:
//   nothing 
// ********************************************************

void ta_Software_ShowSVG()
{
  string strDisplay = PROJ_PATH + "Data/TopAceOutput/_index.html";
  system( strDisplay );
}

// ********************************************************
// Description:
//   Produce a graph for each of the modules
//
//  This function is (maybe) done after the script
//  files have been analysed. This means that we
//  continue the progress bar that has been started
//  in the previous stepes. That is why we
//   get the argument 'iProgressBarStart'

// Usage:
//   ta_Software_ShowSVG();
//
// Returns:
//   nothing 
// ********************************************************

void ta_Software_ProduceGraphPerModule(
  dyn_string dstrModules,   // The modules to be converted
  shape PROGRESS,           // The shape with the progress bar
  int iProgressBarStart,    // 0 or an offset when script files have been analysed
  bool bDoMetrics           // TRUE -> add the metrics
)
{

  
  for( int t = 1; t <= dynlen( dstrModules ); t++)
  {
    string strModule = dstrModules[t];
    
    DebugN( "Producing graph/html for module : " + strModule );
    
    // We produce the graph and the svg for each individual module
    ta_Software_ProduceGraphForModule( strModule, dstrModules, bDoMetrics );
    
    PROGRESS.progress( iProgressBarStart + t );
  }
  

}


// ********************************************************
// Description:
//   Produce the graph for one(!) module.
//   Means that we just display:
//   - the modules that use this module
//   - the modules used by this module
//
// Usage:
//   ta_Software_ProduceGraphForModule( "debugging" );
//
// Returns:
//   nothing 
// ********************************************************

ta_Software_ProduceGraphForModule(
  string strModule,         // The module for which we produce a graph    
  dyn_string dstrModules,
  bool bDoMetrics           // true -> add the metrics
)
{
  dyn_string dstrShowModules;
  dyn_string dstrTop3;
  dyn_string dstrTop3Colors;
  string strColor;
  dyn_dyn_string ddstrDummy;  
  
  // Get a reference to the module
  shared_ptr<ta_Module> pModule = g_ta_GlobalData.GetModule( strModule );

 
  
  // Determine the top3 colors
  // for the 'worst' modules
  ta_Software_DetermineScoreTop3(
    strModule,  
    pModule.GetUsedBy(),        // The first set of modulesl
    pModule.GetUses(),          // second set of modules
    dstrTop3,                   // Receives the top 3 of bad modules
    dstrTop3Colors );           // receives the colors for the top 3 bad modules
  
  // Produce the graph for this module
  // and then tell DOT to turn it into a SVG

  
  // Start with an empty graph
  string strGraph = "digraph software_overview{\n";
  
  // we want the module itself to be presented by a box
  strGraph += strModule + "[shape=box " + ta_Software_GetScoreColor( strModule, dstrTop3, dstrTop3Colors ) + "];\n";

  
  // First display all the modules that use us
  
  // =========================================================================
  // What modules use this module
  dyn_string dstrModulesUseUs = pModule.GetUsedBy();
    
  for( int i = 1; i <= dynlen( dstrModulesUseUs ); i++)
  {
    string strModuleUsesUs = dstrModulesUseUs[i];
    
    string strColor = ta_Software_GetScoreColor( strModuleUsesUs, dstrTop3, dstrTop3Colors );
    if( strlen( strColor ))
    {
      strGraph += strModuleUsesUs + "[" + strColor + "];\n";
    }
      
    bool bRelatieStevig; 
    int iNumber;  
      
    pModule.GetUsedByModuleProps(  strModuleUsesUs, bRelatieStevig, iNumber );   
    
    string strConnectie;
      
    // When a module is used more than 1 times
    // then we add a label to the line
    string strLabel;
      
    if( iNumber > 1 )
    {
      strLabel = "[label=" + iNumber + "]";
    }
     
    if( bRelatieStevig )
    {
      // De relatie is stevig en dus tekenen wen een normale lijn
      strConnectie = "  " + strModuleUsesUs + " -> " + strModule + strLabel + ";\n";
    }
    else
    {
      // De relatie is niet stevig en dus tekenen we
      // een gestippelde lijn
      strConnectie = "  " + strModuleUsesUs + " -> " + strModule + strLabel + "[style=dotted];\n";
    }  
      
    strGraph += strConnectie;
  }


  // =========================================================================
  // What modules use this module
  dyn_string dstrModulesWeUse =  pModule.GetUses();
    
  for( int i = 1; i <= dynlen( dstrModulesWeUse ); i++)
  {
    string strModuleWeUse = dstrModulesWeUse[i];
      
    string strColor = ta_Software_GetScoreColor( strModuleWeUse, dstrTop3, dstrTop3Colors );
    if( strlen( strColor ))
    {
      strGraph += strModuleWeUse + "[" + strColor + "];\n";
    }
    
    bool bRelatieStevig ; // = g_ta_Software_mModules[ "modules" ][ strModule ][ "gebruiktmodule" ][ strModuleWeUse ][ "strength" ];
    int iNumber         ; // = g_ta_Software_mModules[ "modules" ][ strModule ][ "gebruiktmodule" ][ strModuleWeUse ][ "number" ];    
      
    pModule.GetUsesModuleProps( strModuleWeUse, bRelatieStevig, iNumber );
    
    string strConnectie;
      
    // When a module is used more than 1 times
    // then we add a label to the line
    string strLabel;
      
    if( iNumber > 1 )
    {
      strLabel = "[label=" + iNumber + "]";
    }
     
    if( bRelatieStevig )
    {
      // De relatie is stevig en dus tekenen wen een normale lijn
      strConnectie = "  " + strModule + "-> " + strModuleWeUse + strLabel + ";\n";
    }
    else
    {
      // De relatie is niet stevig en dus tekenen we
      // een gestippelde lijn
      strConnectie = "  " + strModule + "-> " + strModuleWeUse + strLabel + "[style=dotted];\n";
    }  
      
    strGraph += strConnectie;
  }

  strGraph += "}\n";  
  
  // Now store the graph
  string strFile = PROJ_PATH + "Data/TopAceOutput/Graph/" + strModule + ".txt";
  
  file f = fopen( strFile, "w" );
  fputs( strGraph, f );
  fclose( f );
  
  ta_Software_ProduceSVG( strFile );
  
  dynAppend( dstrShowModules, strModule );
  dynAppend( dstrShowModules, pModule.GetUsedBy() );
  dynAppend( dstrShowModules, pModule.GetUses() );


  
  // Now produce the HTML file
  ta_HTML_ProduceHTML( strModule, dstrShowModules, true, true, bDoMetrics, false, "", "",  ddstrDummy );
}


    
void ta_Software_AnalyseFiles( 
 dyn_string dstrModules, 
 shape PROGRESS 
)
{
  ta_ScriptCode_AnalyseFiles( dstrModules, PROGRESS );
}

// ********************************************************
// Description:
//   Look at the given module, sort by their order
//   and then determine the top3
//
// Usage:
//   ta_Software_ProduceGraphForModule( "debugging" );
//
// Returns:
//   nothing 
// ********************************************************

void ta_Software_DetermineScoreTop3(
  string strModuleBeingDrawn,       // What module are we drawing (of an empty string)
  dyn_string dstrModules1,          // The first set of modulesl
  dyn_string dstrModules2,          // second set of modules
  dyn_string &dstrTop3,             // Receives the top 3 of bad modules
  dyn_string &dstrTop3Colors        // receives the colors for the top 3 bad modules
)
{
  dyn_dyn_anytype ddaSort;
  dyn_string dstrColors = makeDynString( "#FF0000", "#FF6A00", "#FFD800" );
  

  // Turn into one dyn_string
  dynAppend( dstrModules1, dstrModules2 );
  
  // Also add the module being drawn itself
  if( strlen( strModuleBeingDrawn ))
  {
    dynAppend( dstrModules1, strModuleBeingDrawn );
  }  
  
  // Now collect the score for each of the modules
  for(int t = 1; t <= dynlen( dstrModules1 ); t++)
  {
    string strModule = dstrModules1[t];
    
    // Now get the score and then add everything to a dyn-dyn so that we can sort
    shared_ptr<ta_Module> pModule = g_ta_GlobalData.GetModule( dstrModules1[t] );
       
    int iScore = pModule.GetScore();
    
    dynAppend( ddaSort, makeDynAnytype( dstrModules1[t], iScore ));
  }

  
  // Now sort on the score
  dynDynSort( ddaSort, 2 );

  // we now have an array with the worst at the top
  int t = 1;
  int iScore = 0;
  int iColor = 1;
  int iPreviousScore = 0;
  
  do{
    
    // Now get module and score
    string strModule = ddaSort[t][1];
    iScore       = ddaSort[t][2];

    if( iScore < 0 )
    {
      // Yes, this one has a bad(!) score
      // Now assign a color

      if(  iPreviousScore && ( iPreviousScore != iScore ))
      {
        // When the score is different
        // For example: when you step from -2 to -1
        // then we take the next color
        iColor++;
      }    
      
      iPreviousScore = iScore;
      
      // we now know what color to use
      dynAppend( dstrTop3       , strModule );
      dynAppend( dstrTop3Colors , dstrColors[ iColor ] );
    }    

    // We can stop when we reach the end
    // or when we see a module with a perfect socre (0)    
    t++;
    
  }while( (t <= dynlen( ddaSort)) && (iScore < 0 ) && (iColor <= 3) );
}


string ta_Software_GetScoreColor(
  string strModule,                 // For what module are we looking for a color    
  dyn_string dstrTop3,             // Receives the top 3 of bad modules
  dyn_string dstrTop3Colors        // receives the colors for the top 3 bad modules
)
{
  int iPos = dynContains( dstrTop3, strModule );
  
  if( iPos > 0 )
  {
    return " style=filled fontcolor=\"#FFFFFF\" fillcolor=\"" + dstrTop3Colors[ iPos ] + "\"";
  }
  
  return "";
}


