//*********************************************************
// Copyright  2018 by FM . Please copy and distribute. "Share the fun"
//*********************************************************
// Description:
//  This library contains the functions to produce the HTML
//  output for TopAce.
//
//  The HTML pages contain 'embedded svg' where the 
//  SVG is produced by DOT ( See https://www.graphviz.org/ )
//
//  You can find the HTML in the folder \data\TopAceOutput
//
// ********************************************************

//module:TopAce_HTMLOutput
//prefix:ta_HTML_


// ********************************************************
// Description:
//  Will produce the HTML file for one particular
//  module.
//  
//
// Usage:
//   ta_HTML_ProduceHTML( "main", dstrModules );
//
// Returns:
//   Nothing
//
// ********************************************************

void ta_HTML_ProduceHTML(
  string strModule,             // The module for which we produce a HTML page    
  dyn_string dstrModules,       // A list with all modules
  bool bShowModules = false,    // TRUE -> Show a list with all the modules
  bool bShowFiles   = false,    // TRUE -> show all the files
  bool bDoMetrics   = false,    // TRUE -> add the metrics per file
  bool bShowDiff    = false,    // show the modified files
  string strHTMLTemplate = "",  // What template to use
  string strTitle = "",         // what title to show
  dyn_dyn_string ddstrDiff

)
{
  string strSVG;
  
  DebugN( "Producing HTML for module : " + strModule );
  
  // Where is the SVG
  string strSVGFile = PROJ_PATH + "Data/TopAceOutput/Graph/" + strModule + ".txt.svg";
  fileToString(strSVGFile, strSVG );
  
  // The SVG will be something like:
  // <?xml version="1.0" encoding="UTF-8" standalone="no"?>
  // <!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN"  "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
  // <!-- Generated by graphviz version 2.38.0 (20140413.2041)
  // we want to remove the part before the first "<!--
  int iPos = strpos( strSVG, "<!--" );
  if( iPos > 0 )
  {
    strSVG = substr( strSVG, iPos );
  }
  
  ta_HTML_ShapeToHyperlink( strSVG, dstrModules );
  
  // Get the template
  string strTemplate = ta_HTML_GetHTMLTemplate( strHTMLTemplate );
  
  // Insert the modulename into the title
  strreplace( strTemplate, "MODULENAME", strModule );
  
  // Now 'inject' the SVG into this HTML
  strreplace( strTemplate, "<!-- SVG -->", strSVG );
  
  if( strlen( strTitle ))
  {
    // A title was given so use
    // it at the top of the HTML page
    strreplace( strTemplate, "<!-- TITLE -->", strTitle );
  }  
  
  if( bShowModules )
  {
    // Yes, they want to see a list with modules
    strreplace( strTemplate, "<!-- MODULES -->", ta_HTML_FormatTableModules( dstrModules, bDoMetrics ));
  }  
  
  if( bShowFiles )
  {
    // Yes, they want to see a list with modules
    strreplace( strTemplate, "<!-- FILES -->", ta_HTML_FormatFiles( strModule, bDoMetrics ));
  }

  if( bShowDiff )
  {
    // Yes, they want to see a list of the modified files
    strreplace( strTemplate, "<!-- MODIFIED -->", ta_HTML_FormatDiff( ddstrDiff  ));
  }  
  
  // Where do we put the HTML
  string strHTML = PROJ_PATH + "Data/TopAceOutput/" + strModule + ".HTML";
  
  file f =fopen( strHTML, "w" );
  fputs( strTemplate, f );
  fclose( f );
  
}


// ********************************************************
// Description:
//  We use an HTML template for a HTML page
//  with an inline SVG
//
//  The template can be found in:
//   Data/TopAceOutput/template/Inline_svg.html
//
// Usage:
//   ta_HTML_GetHTMLTemplate();
//
// Returns:
//   The contents of the entire HTML template
//
// ********************************************************

string ta_HTML_GetHTMLTemplate(
  string strTemplate = ""       // Overrule this parameter to take a different template
)
{
  string strResult;
  
  // When no template is given
  // then we use the default HTMNL claled 'Template.html'
  if( strTemplate == "" )
  {
    strTemplate = "Template";
  }
  
  // Where is the template
  string strFile = PROJ_PATH + "Data/TopAceOutput/Template/" + strTemplate + ".html";
  
  fileToString( strFile, strResult );
  
  // Retun the contents of the entire HTMl template
  return strResult;
  
}


// ********************************************************
// Description:
// An SVG file will be something like:
//
//    <!-- inloggen -->
//    <g id="node1" class="node"><title>inloggen</title>
//    <ellipse fill="none" stroke="black" cx="40.9464" cy="-90" rx="40.8928" ry="18"/>
//    <text text-anchor="middle" x="40.9464" y="-86.3" font-family="Times New Roman,serif" font-size="14.00">inloggen</text>
//    </g>
//
// We look for the lines that look like "<!-- inloggen -->"
// and turn such a section into a hyperlink
//
// Usage:
//   ta_HTML_ShapeToHyperlink( strSVG, dstrModules );
//
// Returns:
//  nothing (injected directly into the SVG code !)
//
// ********************************************************


void ta_HTML_ShapeToHyperlink(
  string &strSVG,             // The SVG where we turn shapes into hyperlinks
  dyn_string dstrModules      // The list with all modules
)
{
  // we need separate lines
  dyn_string dstrLines = strsplit( strSVG, "\n" );
  
  
  for( int t = 1; t <= dynlen( dstrModules ); t++)
  {
    string strModule = dstrModules[t];
    
    // We do a quik search. Can we find the line that looks like  "<!-- inloggen -->"
    string strSearch = "<!-- " + strModule + " -->";
   
    if( strpos( strSVG, strSearch ) > 0 )
    {
      
      // Yes, we can find a reference to this module
      // now, we need to turn it into a hyperlinl
      // Means that we first need to find this line
      
      bool bFound = false;
      for( int i = 1; (i <= dynlen( dstrLines )) && !bFound; i++)
      {
        string strLine = dstrLines[i];
        
        if( strpos( strLine, strSearch ) >= 0 )
        {
          // yes, we found the line !!!
          // Now add a hyperlink
          dynInsertAt( dstrLines, "<a href=\"" + strModule + ".html\">", i + 1);
                       
          // Now look for the end of this shape
          // skip the current and the line we just added
          i += 2;
          
          // Now we look for the next shape that starts with "<!--" or until we see </svg>
          do{
            strLine = dstrLines[i];
            
            if( (strpos( strLine, "<!--" ) == 0 ) || (strpos( strLine, "</svg>") >= 0 ))
            {
              // Yes, we found the proper line
              dynInsertAt( dstrLines, "</a>", i );
                           
              bFound = true;
            }
            else
            {
              i++;
            }
          }while( !bFound );
          
        }
      }
    }
  }
  
  // Now turn the dyn_ back into a normal string
  strSVG = "";
  for( int t = 1; t <= dynlen( dstrLines ); t++)
  {
    strSVG += dstrLines[t] + "\n";
  }
}

// ********************************************************
// Description:
// The HTML page can ocntain a list with the modules
// in a project.
// Each module is a hyperlink so that they can navigate to that particular
// page
//
//
// Usage:
//  string strExtraText = ta_HTML_FormatTableModules( dstrModules );
//
// Returns:
//  The fully qualified HTML code that is 
//  added to display a table with modules
//
// ********************************************************

string ta_HTML_FormatTableModules(
  dyn_string dstrModules,   // The list with modules that we want to show    
  bool bDoMetrics           // TRUE -> show the metrics score
)
{
  string strResult;
  
  strResult += "<h1>List of modules</h1><br/>\n";
  strResult += "This application contains the following modules:<br/><br/>\n";
  
  strResult += "<table border=\"1\">\n";
  for( int t = 1; t <= dynlen( dstrModules ); t++)
  {
    string strModule = dstrModules[t];
    strResult += "<tr><td><a href=\"" + strModule + ".html\">" + strModule + "</a></td>";
    
    if( bDoMetrics )
    {
      shared_ptr<ta_Module> pModule = g_ta_GlobalData.GetModule( strModule );
      int iScore = pModule.GetScore();
      strResult += "<td>" + iScore + "</td>";
    }
    
    strResult += "</tr>\n";
  }
  strResult += "</table>\n";
  strResult += "<br/";
  
  return strResult;
}

string ta_HTML_FormatFiles(
  string strModule,            // name of the module    
  bool bDoMetrics              // TRUE -> add the metrics per file
)
{
  string strResult;

  // Get a reference to the modules  
  shared_ptr<ta_Module> pModule = g_ta_GlobalData.GetModule( strModule );

  
  strResult += "<h1>List of files</h1><br/>\n";
  strResult += "This modules contains the following files:<br/><br/>\n";
  
  dyn_string dstrFiles = pModule.GetFileNames();
  
  strResult += "<table border=\"1\">\n";
  strResult += "<tr>";
  strResult += "<th>Name</th>";
  
  if( bDoMetrics )
  {
    strResult += "<th>Score</th>";
  }
  
  strResult += "<th>Size</th>";
  strResult += "<th>Date/time</th>";
  
  // when they want to see the metrics
  // then add some columns to the table
  if( bDoMetrics )
  {
    strResult += "<td>#lines</td>";
    strResult += "<td>#code</td>";
    strResult += "<td>white/code</td>";
    strResult += "<td>c/code</td>";
    strResult += "<td>McCabe</td>";
    strResult += "<td>McCabe/Code</td>";
  }
  
  strResult += "</tr>\n";
  
  for( int t = 1; t <= dynlen( dstrFiles ); t++)
  {
    string strFile = dstrFiles[t];
    
    // Get a reference to the 
    shared_ptr<ta_File> pFile = pModule.GetFile(strFile );
        
    int iSize = pFile.GetSize();
    string strStamp = pFile.GetStampString();
    
    strResult += "<tr>";
    strResult += "<td>" + strFile + "</td>";
    
    if( bDoMetrics )
    {
      strResult += "<td>" + pFile.GetScore() + "</td>";
    }
    
    strResult += "<td>" + iSize + "</td>";
    strResult += "<td>" + strStamp + "</td>";
    
    
    if( bDoMetrics )
    {
      string strWhitePerLineCode;
      string strMcCabePerLineCode;
      string strBytesCommentPerByteCode;
      
     
      // When a certain condition is bad, then we want the table cell to be red
      string strCellWhiteLines          = "<td>";
      string strCellCommentsPerByteCode = "<td>";
      string strCellMcCabePerLinesCode  = "<td>";
      
      if( pFile.HasBadScore_WhiteLines()         ) strCellWhiteLines          = "<td bgcolor=\"FF0000\">";
      if( pFile.HasBadScore_ByteCommentPerCode() ) strCellCommentsPerByteCode = "<td bgcolor=\"FF0000\">";
      if( pFile.HasBadScore_McCabePerLine()      ) strCellMcCabePerLinesCode  = "<td bgcolor=\"FF0000\">";
      
      strResult += "<td>" + pFile.GetProp( "numlines" ) + "</td>";
      strResult += "<td>" + pFile.GetProp( "reallinescode" ) + "</td>";
      strResult += strCellWhiteLines+ pFile.GetScore_WhiteLines() + "</td>";
      strResult +=strCellCommentsPerByteCode + pFile.GetScore_Comments() + "</td>";
      strResult += "<td>" + pFile.GetProp( "mccabe" ) + "</td>";
      strResult +=strCellMcCabePerLinesCode + pFile.GetScore_McCabe() + "</td>";
    }
    
    strResult += "</tr>\n";
    
    // when they want to see metrics and we have some reasons to show
    // then add those reasons to the file now:
    dyn_string dstrReasons = pFile.GetReasonsBad();
    
    if( dynlen( dstrReasons ) && bDoMetrics )
    {
      // Yes, we have don emetrics and there are reaons why this file is bad
      // so lets add the reasons to the table
      for( int i = 1; i <= dynlen( dstrReasons ); i++)
      {
         strResult += "<tr><td></td><td></td>";
         strResult += "<td colspan=\"8\">" + dstrReasons[i] + "</td></tr>";
      }
    }
  }
  strResult += "</table>\n";
  strResult += "<br/";
  
  return strResult;
  
}

// ********************************************************
// Description:
// The HTML page can contain a section with the modified
// files.
//
// Usage:
//  string strExtraText = ta_HTML_FormatDiff( ddstrDiff );
//
// Returns:
//  The fully qualified HTML code that is 
//  added to display a table withthe modified files
//
// ********************************************************

string ta_HTML_FormatDiff(
  dyn_dyn_string ddstrDiff // what files have been modified    
 )
{
  string strResult;
  
  strResult += "<h1>Modified files</h1><br/>\n";
  strResult += "The following files have been modified/added since the version.<br/><br/>\n";
  
  strResult += "<table border=\"1\">\n";
  strResult += "<tr>";
  strResult += "<th>Module</th>";
  strResult += "<th>File/th>";
  strResult += "</tr>";
  
  for( int t = 1; t <= dynlen( ddstrDiff ); t++)
  {
    strResult += "<tr>";
    strResult += "<td>" + ddstrDiff[t][1] + "</td>";
    strResult += "<td>" + ddstrDiff[t][2] + "</td>";
    strResult += "</tr>";
  }

  strResult += "</table>\n";
  strResult += "<br/>";

  return strResult;  
}
