//*********************************************************
// Copyright  2018 by FM . Please copy and distribute. "Share the fun"
//*********************************************************
// Description:
//  This file contains some classes that hold the global
//  data for the TopAce application.
//  This makes it possible to define clean 'put' and 'get'
//  methods shielding of the details of internal data.
//
//  The top object is the 'ta_GlobalData'. When we scan the 
//  source and find a 'moduke' then a ta_Module class is 
//  added.
//  A ta_Module contains one or more ta_Files.
//
//  (1) ta_GlobalData (1)-> (N) ta_Module (1) -> (N) ta_File
//
//  NOTE, NOTE, NOTE, NOTE, NOITE, NOTE, NOTE !!!!
//  ==============================================
//  Note: There were issues with these classes and especially
//  the usage of the 'shared_ptr'. The issues were solved
//  when using 3.15P9
//
// ********************************************************

//module:TopAce_GlobalData
//prefix:ta_GlobalData

// The following constants define the 'limits' 
// when does TopAce find a file 'bad'
// Note: Use the menu-item "TopAce\Analyse script folder"
// to analyse your project and to figure out your values.
// What libraries do you consider good or bad
const float TA_GLOBAL_MIN_COMMENT_PER_CODE    = 0.5;   // Minimum ratio comment to code
const float TA_GLOBAL_MAX_MCCABE_PER_CODE     = 0.3;   // Maximum index. McCabe per code
const float TA_GLOBAL_MIN_WHITELINES_PER_CODE = 0.3;   // Minimum ratio #white lines to #lines code (excluding comments)

// we want a minimum number of source lines
// before we do an analsys
const int TA_GLOBAL_MIN_NUMLINES = 30;

// ********************************************************
// Description:
//   The ta_File is a class to hold the name and properties
//   of one file.
//   The name of the file is a relative file
//   ( like "scripts/libs/test_Dummy.ctl" )
//
//   The class holds:
//   - size of file
//   - date/time of file
//   - various calculated properties ( like McCabe index )
//
// ******************************************************** 
class ta_File{
  
  // ********************************************************
  // Description:
  //   This is the default constructor for the ta_file
  //   class.
  //   Will store the name of the file and will also
  //   collect the filesize and the stamp of the file
  //
  // Usage:
  //   ta_File f = ta_File( "scripts/libs/test_Dummy.ctl" );
  //
  // Returns:
  //   Nothing
  // ********************************************************  
  public ta_File(
    string strName,
    string strOverrideAbsoluteFile = ""  
  )                                      
  {
    strFileName = strName;
    
    // Now get some initial information about this file
    string strAbsoluteFile;
   
    if( strOverrideAbsoluteFile == "" )
    {
      strAbsoluteFile = GetAbsoluteFile();
    }
    else
    {
      strAbsoluteFile = strOverrideAbsoluteFile;  
    }  

    // Get the size of the file and the modification stamp of the file
    iFileSize  = getFileSize( strAbsoluteFile );
    tFileStamp = getFileModificationTime( strAbsoluteFile );
    
  }

  // ********************************************************
  // Description:
  //  Various simple 'get' methods that give controlled
  //  acess to the private members of this class
  //
  // Usage:
  //   ..
  //
  // Returns:
  //   Nothing
  // ********************************************************   
  public string GetName()        { return strFileName; }         // Get the relative name of the file (add PROJ_PATH to get absolute)
  public string GetSize()        { return iFileSize; }           // get the size of the file
  public string GetStamp()       { return tFileStamp; }          // get the modification date/time of the file
  public string GetStampString() { return (string)tFileStamp; }  // Get the modofication date/time converted to a string
  
  public bool ScriptFile() { return strpos( strFileName, ".ctl" ) > 0; }
  public bool PanelFile()  { return strpos( strFileName, ".pnl" ) > 0; }
    
    
  // ********************************************************
  // Description:
  //  The ta_File holds a relative filename ( e.g. "scripts/libs/test_Dummy.ctl" ).
  //  This function is used to determine the absolute file name
  //
  // Usage:
  //   string strAbsolyteFile = f.GetAbsoluteFile();
  //
  // Returns:
  //   An absolute reference to a file
  // ********************************************************  
  
  public string GetAbsoluteFile() 
  { 
    return PROJ_PATH + strFileName; 
  }

  // ********************************************************
  // Description:
  //  A ta_File can have a set of properties.
  //  Typically determined when doing an analysis of a script.
  //  This method is used to store the properties that
  //  have been found
  //
  // Usage:
  //   mapping m;
  //   m[ "size" ] = 1000;
  //   f.SetProps( m );
  //
  // Returns:
  //   nothing
  // ********************************************************    
  public void SetProps( mapping &m )
  {
    mProps = m;
  }
  

  // ********************************************************
  // Description:
  //  A safe method to get one of the props of a ta_File.
  //  Will check the mapping to make sure that the prop
  //  actually exists
  //
  // Usage:
  //  string strValue = f.GetProp( "numlines" );
  //
  // Returns:
  //   The property
  // ******************************************************** 
  
  public anytype GetProp( string strProp )
  {
    if( mappingHasKey( mProps, strProp ))
    {
      return mProps[ strProp ];
    }
    
    return "-";
  }  
  
  public bool HasBadScore_WhiteLines()
  {
    if( EnoughSourceLines() )
    {
      return GetProp( "whitelinesperlinecode"    ) < TA_GLOBAL_MIN_WHITELINES_PER_CODE ;
    }
  
    return false;  
  }
  public bool HasBadScore_ByteCommentPerCode()
  {
    if( EnoughSourceLines() )
    {
      return GetProp( "bytescommentperbytescode" ) < TA_GLOBAL_MIN_COMMENT_PER_CODE;
    }
    return false;
  }
  
  public bool HasBadScore_McCabePerLine()
  {
    if( EnoughSourceLines() )
    {
      return GetProp( "mccabeperlinescode"       ) > TA_GLOBAL_MAX_MCCABE_PER_CODE ;
    }
    return false;  
  }
  
  public string GetScore_WhiteLines()
  {
    string strResult;
    
    if( EnoughSourceLines() )
    {
      // Yes, we have enough lines of source to 
      // do show the analysis results
      sprintf( strResult, "%1.2f",  GetProp( "whitelinesperlinecode"    ) );
      return strResult;    
    }
    
    return "-";
  }
  
  public string GetScore_Comments()
  {
    bool bDebug = false;
    string strResult;
    
    string strName = GetName();
    if( strpos( strName, "Analyse_ScriptCode.pnl" ) >= 0 )
    {
      bDebug = true;
      
    }
    
    if( EnoughSourceLines() )
    {
      // Yes, we have enough lines of source to 
      // do show the analysis results
      sprintf( strResult, "%1.2f",  GetProp( "bytescommentperbytescode" ) );
      
      if( bDebug ) DebugN( "<==== " + strResult + " bytes=" + GetProp( "bytescommentperbytescode" ) );

      return strResult;    
    }

    if( bDebug ) DebugN( "======= NOT ENOUGH LINES" );
    
    return "-";
  }  
  
  public string GetScore_McCabe()
  {
    string strResult;
    
    if( EnoughSourceLines() )
    {
      // Yes, we have enough lines of source to 
      // do show the analysis results
      sprintf( strResult, "%1.2f",  GetProp( "mccabeperlinescode" ) );
      return strResult;    
    }
    
    return "-";
  }  
  // ********************************************************
  // Description:
  //  Is used to get the total number of source
  //  lines. Especially needed when we do 
  //  an analysis of source code
  //
  //  we do not want to any anyslis
  //  when a panel only has a couple of lines
  //
  // Usage:
  //  int iLines = f.GetTotalNumLines();
  //
  // Returns:
  //   The clean number of source lines
  //   (extracted from a panel)
  // ******************************************************** 
    
  public int GetTotalNumLines()
  {
    // Get the number of lines of source coe
    int iLines = GetProp( "numlines" );
   
    return iLines; 
  }
  
  public int GetWhiteLinesCode()
  {
    return GetProp( "whitelinescode" );
  }

  public int GetRealLinesCode()
  {
    return GetProp( "reallinescode" );
  }
  
  public int GetMcCabe()
  {
    return GetProp( "mccabe" );
  }
  
  // ********************************************************
  // Description:
  //  Is used to verify that we have enough
  //  lines of source code to do a proper
  //  analysis. 
  //  Example: You can't say much
  //  about number of ocmments when the whole source
  //  is just 10 lines of source code
  //
  // Usage:
  //  bool bEnough = f.EnoughSourceLines();
  //
  // Returns:
  //   TRUE when there are enough lines of source code
  //   to do a TopAce analysis
  // ******************************************************** 
  
  public bool EnoughSourceLines()
  {
    // do we have enough lines of source code
    // to do a proper analysis
    return GetTotalNumLines() > TA_GLOBAL_MIN_NUMLINES;
  }
  // ********************************************************
  // Description:
  //  Method 'Determinescore()' is used to see how
  //  good this source file isa ccording to TopAce
  //
  //  This method can be used to read that score
  //
  // Usage:
  //  int iScore = f.DetermineScore();
  //
  // Returns:
  //   The score. a negative number that shows how 
  //   'bad' the source file is
  // ******************************************************** 
  
  public int DetermineScore()
  {
    dstrReasonBadScore = makeDynString();
    
    // Start a score of 0
    iScore = 0;
    
    float fWhiteLinesPerLineCode   = GetProp( "whitelinesperlinecode"    );
    float fBytesCommentPerByteCode = GetProp( "bytescommentperbytescode" );
    float fMcCabePerLinesCode      = GetProp( "mccabeperlinescode"       );
    
    // Determine the score for individual criteria
    if( HasBadScore_WhiteLines()         )
    {
      dynAppend( dstrReasonBadScore, "#White_lines/#lines_code is under " + TA_GLOBAL_MIN_WHITELINES_PER_CODE);
      iScore--;
    }  
    if( HasBadScore_ByteCommentPerCode() )
    {
      dynAppend( dstrReasonBadScore, "#bytes_comment/#bytes_code is under " + TA_GLOBAL_MIN_COMMENT_PER_CODE);
      iScore--;
    }  
    if( HasBadScore_McCabePerLine()      )
    { 
      dynAppend( dstrReasonBadScore, "#McCabe/#lines_code is over " + TA_GLOBAL_MAX_MCCABE_PER_CODE);
      iScore--;
    }  
    
    return iScore;
  }
  
  // ********************************************************
  // Description:
  //  Method 'Determinescore()' has determined the score for this
  //  file and has also build an array with 'reasons'
  //
  //  This method can be used to read that array
  //
  // Usage:
  //  dyn-string dstrReasons = f.GetReasonsBad()
  //
  // Returns:
  //   an array wih reasons why the score is bad
  // ******************************************************** 
  
  public dyn_string GetReasonsBad()
  {
    return dstrReasonBadScore;
  }
  
  // ********************************************************
  // Description:
  //  Method 'Determinescore()' is used to see how
  //  good this source file isa according to TopAce
  //
  //  This method can be used to read that score
  //
  // Usage:
  //  int iScore = f.GetScore();
  //
  // Returns:
  //   The score. a negative number that shows how 
  //   'bad' the source file is
  // ******************************************************** 
  
  public int GetScore()
  {
    return iScore;
  }
  
  // ********************************************************
  // Description:
  //  The button 'Label' in the TopAce window
  //  is used to define a version. This means that we
  //  record all properties of a version and store this in
  //  a csv file.
  //
  //  This method will go through our model of
  //  modules and files adding them all to the given
  //  dyn_dyn
  //
  // Usage:
  //  dyn_dyn_string ddstrVersionInfo;
  //  g_ta_GlobalData.AddVersionInfo( ddstrVersionInfo );
  //
  // Returns:
  //  Nothing. Everything is returned via the argument
  //  ddstrVersionInfo
  // ******************************************************** 
  
  public void AddVersionInfo( 
    string strModule,                   // The mopdule that we are part of  
    dyn_dyn_string &ddstrVersionInfo    // will receive the version info
  )
  {
    // Add a 'record' describing our properties
    dyn_string dstrRecord = makeDynString(
      strModule,
      strFileName,
      (string)tFileStamp,
       iFileSize 
    );

    dynAppend( ddstrVersionInfo, dstrRecord );
  }
  
  
  // Data members ( all private)
  private string     strFileName;        // The relative name of the file (add PROJ_PATH to get an absolute)
  private int        iFileSize;          // The size of the file
  private time       tFileStamp;         // The stamp of the file
  private mapping    mProps;             // Custom properties determined for a file
  private int        iScore;             // The TopAce score for this file
  private dyn_string dstrReasonBadScore; // Reasons why the score is bad
};

// ********************************************************
// Description:
//  The ta_Module holds the properties for one module.
//  A module is basicaly a soet of (ta_File) files
//  and relationships to other modules
//
//
// ********************************************************
class ta_Module{
  
  // ********************************************************
  // Description:
  //   This is the default constructor for the ta_Module
  //   class.
  //   We just store the name of the module here
  //
  // Usage:
  //   ta_Module m = ta_Module( "alarms" );
  //
  // Returns:
  //   Nothing
  // ********************************************************
  public ta_Module( string strName )
  {
    strModuleName = strName;
  }

  // ********************************************************
  // Description:
  //   Will return the name of the module
  //
  // Usage:
  //   string strName = m.GetName();
  //
  // Returns:
  //   The name of the module
  // ********************************************************  
  public string GetName() { return strModuleName; }
  
  // ********************************************************
  // Description:
  //  Is used to add a filename to a module
  //  E.g. is called when we find a file that 
  //  belongs to a certain module.
  //
  //  Will first verify that this file is 
  //  really new (will only add when we don;t yet have the file)
  //
  // Usage:
  //   m.AddFile( "c:/tmp/test.txt" );
  //
  // Returns:
  //   a pointer to the existing or newly added file
  // ********************************************************    
  public shared_ptr<ta_File> AddFile( string strFile )
  {
   
    if( GetFile( strFile ) == nullptr )
    {
      // This file is new so add to our list
      mFiles[ strFile ] = new ta_File( strFile );    
    }  
    
    return GetFile( strFile );
  }
  
  // ********************************************************
  // Description:
  //  Can be used to locate the ta_file given the relative
  //  name of the file
  //
  // Usage:
  //  shared_ptr<ta_File> pf = m.GetFile( "c:/tmp/test.txt" );
  //
  // Returns:
  //   A shared ptr to a ta_file or a nullptr
  //   when no match was found
  // ********************************************************   
  public shared_ptr<ta_File> GetFile( string strFile )
  {
    if( mappingHasKey( mFiles, strFile ))
    {
      return mFiles[ strFile ];
    }
    else
      return nullptr;
  }
  
  // ********************************************************
  // Description:
  //  Is used to check wether a certain file exists
  //  in this module
  //  
  //
  // Usage:
  //  bool bexists = m.HasFile( strFile );
  //
  // Returns:
  //   true when the specified file has been found
  // ******************************************************** 
  
  public bool HasFile( string strFile )
  {
    dyn_string dstrFiles = mappingKeys( mFiles );    
    return dynContains( dstrFiles, strFile ) > 0;
  }
  
  // ********************************************************
  // Description:
  // Can used to ask for the number of files
  // that are part of a module
  //
  // Usage:
  //  int iNumFiles = m.GetNumFiles();
  //
  // Returns:
  //   The number of files in this module
  // ********************************************************     
  public int GetNumFiles() 
  {
    return dynlen( GetFileNames() ); 
  }
  
  // ********************************************************
  // Description:
  // Is used to get a list of file names that are in 
  // this module
  // Will return a sorted list of relative filenames
  //
  // Usage:
  //  dyn_string dstrFileNames = m.GetFileNames();
  //
  // Returns:
  //   A sorted list with filenames
  // ********************************************************   
  public dyn_string GetFileNames()
  {
    dyn_string dstrFiles = mappingKeys(  mFiles );  
    dynSort( dstrFiles ); 
    return dstrFiles;
  }
  
  
  // ********************************************************
  // Description:
  // Call this method to indicate that a module
  // is using another module
  //
  // wil create a mapping for every module that we use.
  // this mapping holds:
  //   "strength"
  //      true when it is a function call. false when it is a panel
  //
  //   "number"
  //      How often is this relationship used
  //
  // Usage:
  //    m.AddUses( "alarms", true );
  //
  // Returns:
  //   Nothing
  // ********************************************************    
  public void AddUses(
    string strUsedModule,   // The module that we use
     bool bStrength         // The 'strength' for our relationship
  )
  {
    if( !mappingHasKey( mModulesUsed, strUsedModule ))
    {
      mModulesUsed[ strUsedModule ] = makeMapping();
      mModulesUsed[ strUsedModule ][ "strength" ] = false;
      mModulesUsed[ strUsedModule ][ "number" ] = 0;
    }
      
    mModulesUsed[ strUsedModule ][ "strength" ] |= bStrength;  
    mModulesUsed[ strUsedModule ][ "number" ]++;
  }   
 
   // ********************************************************
  // Description:
  // Call this method to get a list of modules are used
  // in this module. 
  //
  // Usage:
  //    dyn_string dstrUses = m.GetUses();
  //
  // Returns:
  //   a sorted list of modules that we use
  // ********************************************************  
  
  public dyn_string GetUses()
  {
    dyn_string dstrUses = mappingKeys( mModulesUsed );
    dynSort( dstrUses );
    return dstrUses;
  }
  
   // ********************************************************
  // Description:
  // We have list of modules that we use. Now, we can get
  // the properties of one such module.
  // 
  // - how strong is the relationship
  // - how many times are we being used
  //
  // Usage:
  //    m.GetUsesModuleProps( strModule, bStrength, iNumber );
  //
  // Returns:
  //   Nothing (is returned via the parameters)
  // ********************************************************    
  public void GetUsesModuleProps(
    string strModule,       // Get the properties of what module
    bool &bStrength,        // receives the strength
    int &iNumber            // receives the number (how often is the relationship being used)
  )
  {
    bStrength = mModulesUsed[ strModule ][ "strength" ];
    iNumber   = mModulesUsed[ strModule ][ "number" ];
  }
  
  // ********************************************************
  // Description:
  // Call this method to indicate that a module
  // is using us (this module)
  //
  // wil create a mapping for every module that uses us:
  //
  // this mapping holds:
  //   "strength"
  //      true when it is a function call. false when it is a panel
  //
  //   "number"
  //      How often is this relationship used
  //
  // Usage:
  //    m.AddUsedBy( "alarms", true );
  //
  // Returns:
  //   Nothing
  // ********************************************************   
  
  public void AddUsedBy(
    string strUsedByModule,   // The module that is using us
     bool bStrength           // The 'strength' for our relationship
  )
  {
    if( !mappingHasKey( mModulesUsingUs, strUsedByModule ))
    {
      mModulesUsingUs[ strUsedByModule ] = makeMapping();
      mModulesUsingUs[ strUsedByModule ][ "strength" ] = false;
      mModulesUsingUs[ strUsedByModule ][ "number" ] = 0;
    }
      
    mModulesUsingUs[ strUsedByModule ][ "strength" ] |= bStrength;  
    mModulesUsingUs[ strUsedByModule ][ "number" ]++;
  }  
  
   // ********************************************************
  // Description:
  // Call this method to get a list of modules that use
  // this module. That is L the modules that have a reference
  // to this module
  //
  // Usage:
  //    dyn_string dstrUsedBy = m.GetUsedBy();
  //
  // Returns:
  //   a sorted list of modules that use us
  // ********************************************************  
  
  public dyn_string GetUsedBy()
  {
    dyn_string dstrUsedByModules = mappingKeys( mModulesUsingUs );
    dynSort( dstrUsedByModules );
    return dstrUsedByModules;
  }  
  
   // ********************************************************
  // Description:
  // We have list of modules that use us. Now, we can get
  // the properties of one such module.
  // 
  // - how strong is the relationship
  // - how many times are we being used
  //
  // Usage:
  //    m.GetUsedByModuleProps( strModule, bStrength, iNumber );
  //
  // Returns:
  //   Nothing (is returned via the parameters)
  // ********************************************************  
  
  public void GetUsedByModuleProps(
    string strModule,       // Get the propertiesof what module
    bool &bStrength,        // receives the strength
    int &iNumber            // receives the number (how often is the relationship being used)
  )
  {
    bStrength = mModulesUsingUs[ strModule ][ "strength" ];
    iNumber   = mModulesUsingUs[ strModule ][ "number" ];
  }
  
  // ********************************************************
  // Description:
  //  Method 'Determinescore()' is used to see how
  //  good this source module is ccording to TopAce
  //
  //  This method can be used to read that score
  //
  // Usage:
  //  int iScore = m.DetermineScore();
  //
  // Returns:
  //   The score. a negative number that shows how 
  //   'bad' the source file is
  // ******************************************************** 
  
  public int DetermineScore()
  {
    // Start a score of 0
    iScore = 0;
    
    dyn_string dstrFiles = GetFileNames();
    
    for( int t = 1; t <= dynlen( dstrFiles ); t++)
    {
      shared_ptr<ta_File>pFile = GetFile( dstrFiles[t] );
      
      iScore += pFile.DetermineScore();
    }
   
    return iScore;
  }
  
  // ********************************************************
  // Description:
  //  The button 'Label' in the TopAce window
  //  is used to define a version. This means that we
  //  record all properties of a version and store this in
  //  a csv file.
  //
  //  This method will go through our model of
  //  modules and files adding them all to the given
  //  dyn_dyn
  //
  // Usage:
  //  dyn_dyn_string ddstrVersionInfo;
  //  g_ta_GlobalData.AddVersionInfo( ddstrVersionInfo );
  //
  // Returns:
  //  Nothing. Everything is returned via the argument
  //  ddstrVersionInfo
  // ******************************************************** 
  
  public void AddVersionInfo( 
    dyn_dyn_string &ddstrVersionInfo    // will receive the version info
  )
  {
    dyn_string dstrFiles = GetFileNames();
    
    for( int t = 1; t <= dynlen( dstrFiles ); t++)
    {
      shared_ptr<ta_File>pFile = GetFile( dstrFiles[t] );
      
      pFile.AddVersionInfo( GetName(), ddstrVersionInfo );
    }
  }  
  
  // ********************************************************
  // Description:
  //  Method 'Determinescore()' is used to see how
  //  good this module is according to TopAce
  //
  //  This method can be used to read that score
  //
  // Usage:
  //  int iScore = m.GetScore();
  //
  // Returns:
  //   The score. a negative number that shows how 
  //   'bad' the source file is
  // ******************************************************** 
  
  public int GetScore()
  {
    return iScore;
  }
  
  // data members (all private)
  private string strModuleName;   // The unique name of this module
  private mapping mFiles;         // The set of files that are in this module
  
  private mapping mModulesUsed;      // What modules are used from this module
  private mapping mModulesUsingUs;   // What modules use this module
  
  private int     iScore;         // The TopAce score for this module
};

// ********************************************************
// Description:
//  This class actually holds all the global data
//  for the TopAce application.
//  It const of modules where every module
//  holds 1 to N (source) files
//
//
// ********************************************************

class ta_GlobalData{
  
  public ta_GlobalData()
  {
  }
  
  // ********************************************************
  // Description:
  //  Is used to clear out the global data
  //  will simply clear the mapping with modules in this 
  //  class
  //
  // Usage:
  //   g.Reset();
  //
  // Returns:
  //   Nothing
  // ******************************************************** 
  public void Reset()
  {
    mappingClear( mModules );
    dynClear( dstrFiles );
    dynClear( dstrModules );
    dynClear( dstrPrefixModules );    // The module that belongs to a certain prefix
    dynClear( dstrPrefix );           // The prefix for this module    
  }  
  
  // ********************************************************
  // Description:
  //  Is used to add a module to our list.
  //
  //  Will first verify that this module is 
  //  really new (will only add when we don't yet have the module)
  //
  // Usage:
  //   g.AddModule( "alarms" );
  //
  // Returns:
  //   A pointer to a new or existing module
  // ********************************************************    
  public shared_ptr<ta_Module> AddModule( string strModule )
  {
    if( GetModule( strModule ) == nullptr )
    {
      // This module is new so add to our list
      mModules[ strModule ] = new ta_Module( strModule );  
    
    }  
    
    // Return a pointer to the module
    return GetModule( strModule );
  }  

  // ********************************************************
  // Description:
  //  Can be used to locate the ta_Module given the name
  //  of that module
  //
  // Usage:
  //  shared_ptr<ta_Module> pm = g.GetModule( "alarms" );
  //
  // Returns:
  //   A shared ptr to a ta_module or a nullptr
  //   when no match was found
  // ********************************************************   
  public shared_ptr<ta_Module> GetModule( string strModule )
  {
  
    if( mappingHasKey( mModules, strModule ))
    {
      
      return mModules[ strModule ];
    }
    else
      return nullptr;
  }
  
  // ********************************************************
  // Description:
  // Is used to get a list of module names that have
  // been stored.
  // Will return a sorted list
  //
  // Usage:
  //  dyn_string dstrModules = g.GetModuleNames();
  //
  // Returns:
  //   A sorted 
  // ********************************************************   
  public dyn_string GetModuleNames()
  {
    dyn_string dstrModules = mappingKeys(  mModules );  
    dynSort( dstrModules ); 
    return dstrModules;
  }
  
  // ********************************************************
  // Description:
  // Is used to get a list of module names that have
  // been stored.
  // Will return a sorted list
  //
  // Usage:
  //  dyn_string dstrModules = g.GetModuleNames();
  //
  // Returns:
  //   A sorted 
  // ********************************************************  
  public void AddPrefixAndModule(
    string strPrefix,     // The prefix (used to find relations between modules)
    string strModule      // teh name of the module
  )
  {
    // is this prefix new
    if( dynContains( dstrPrefix, strPrefix ) <= 0 )
    {
      // This prefix is new
      dynAppend( dstrPrefix, strPrefix );         // Remember the prefix
      dynAppend( dstrPrefixModules, strModule );  // and the module it represents
    }
  }  
  
  // ********************************************************
  // Description:
  // Is used to get the list of prefix and modules
  // from this global class
  //
  // Usage:
  //  GetPrefixAndModule( dstrPrefix, dstrModules );
  //
  // Returns:
  //  nothing (everything via the parameters)
  // ********************************************************  
  public void GetPrefixAndModule(
     dyn_string &dstrReturnPrefix,   // Will receive the list of prefixes
     dyn_string &dstrReturnModules   // Wil receive the list of modules
  )
  {
    dstrReturnPrefix   = dstrPrefix;
    dstrReturnModules = dstrPrefixModules;
  }
  
  // ********************************************************
  // Description:
  //    We keep a global list of a file
  //    and its module.
  //
  // Usage:
  //  AddFileAndModule( "alarms", "test.pnl" );
  //
  // Returns:
  //  nothing (everything via the parameters)
  // ********************************************************   
  public void AddFileAndModule(
    string strFile,           // and a file in the module
    string strModule          // The module
  )
  {
    dynAppend( dstrFiles, strFile );
    dynAppend( dstrModules, strModule );
  }
  
  // ********************************************************
  // Description:
  //    We keep a global list of a file
  //    and its module.
  //    This function will return those lists
  //
  // Usage:
  //  GetFilesAndModules( dstrModules, dstrFIles );
  //
  // Returns:
  //  nothing (everything via the parameters)
  // ********************************************************   
  
  public void GetFilesAndModules(
    dyn_string &dstrReturnModules,          // Receives a set of modules
    dyn_string &dstrReturnFiles             // .. and a set of files
  )
  {
    // Return the total list of files and modules
    dstrReturnModules = dstrModules;
    dstrReturnFiles   = dstrFiles;
  }  
  
  // ********************************************************
  // Description:
  //  Method 'Determinescore()' is used to see how
  //  good this source module is ccording to TopAce
  //
  //  This method can be used to read that score
  //
  // Usage:
  //  int iScore = m.DetermineScore();
  //
  // Returns:
  //   The score. a negative number that shows how 
  //   'bad' the source file is
  // ******************************************************** 
  
  public int DetermineScore()
  {
    // Start a score of 0
    iScore = 0;
    
    dyn_string dstrModules = GetModuleNames();
    
    for( int t = 1; t <= dynlen( dstrModules ); t++)
    {
      shared_ptr<ta_Module>pModule = GetModule( dstrModules[t] );
      
      iScore += pModule.DetermineScore();
    }
    
    return iScore;
  }
  
  // ********************************************************
  // Description:
  //  The button 'Label' in the TopAce window
  //  is used to define a version. This means that we
  //  record all properties of a version and store this in
  //  a csv file.
  //
  //  This method will go through our model of
  //  modules and files adding them all to the given
  //  dyn_dyn
  //
  // Usage:
  //  dyn_dyn_string ddstrVersionInfo;
  //  g_ta_GlobalData.AddVersionInfo( ddstrVersionInfo );
  //
  // Returns:
  //  Nothing. Everything is returned via the argument
  //  ddstrVersionInfo
  // ******************************************************** 
  
  public void AddVersionInfo( 
    dyn_dyn_string &ddstrVersionInfo    // will receive the version info
  )
  {
    dyn_string dstrModules = GetModuleNames();
    
    for( int t = 1; t <= dynlen( dstrModules ); t++)
    {
      shared_ptr<ta_Module>pModule = GetModule( dstrModules[t] );
      
     
      pModule.AddVersionInfo( ddstrVersionInfo );
    }
  }
  
  // ********************************************************
  // Description:
  //  Method 'Determinescore()' is used to see how
  //  good this Topace project is according to TopAce
  //
  //  This method can be used to read that score
  //
  // Usage:
  //  int iScore = globalData.GetScore();
  //
  // Returns:
  //   The score. a negative number that shows how 
  //   'bad' the the total project is
  // ******************************************************** 
  
  public int GetScore()
  {
    return iScore;
  }  
  
  
  public float GetMinCommentsPerCode()
  {
    return TA_GLOBAL_MIN_COMMENT_PER_CODE;
  }

  public float GetMaxMcCabePerLine()
  {
    return TA_GLOBAL_MAX_MCCABE_PER_CODE;
  }  
  
  public float GetMinWhiteLinesPerCode()
  {
    return TA_GLOBAL_MIN_WHITELINES_PER_CODE;
  }  
  
  
  
  // The total set of modules that have been found
  private mapping mModules;
  
  // we store a list with prefix and prefix modules
  private dyn_string dstrPrefixModules;    // The module that belongs to a certain prefix
  private dyn_string dstrPrefix;           // The prefix for this module

  private dyn_string dstrFiles;            // a total set of files
  private dyn_string dstrModules;          // and the module the file belongs to
  
  private int iScore;
};

// This is our actual global data
global ta_GlobalData g_ta_GlobalData;

